%% How to manually set LSPs in QuaDRiGa
%
% This tutorial explains, how to generate a time-series of channel
% coefficients with manual selection of LSPs.

%% Setting general parameters
% We set up some basic parameters such as center frequency, sample density
% and the position of the transmitter.

close all
clear all

set(0,'defaultTextFontSize', 14)    % Set default font size for the plots
set(0,'defaultAxesFontSize', 14)

s = simulation_parameters;          % Basic simulation parameters
s.center_frequency      = 2.185e9;  % Center Frequency
s.sample_density        = 4;        % 4 samples / half wave length

l = layout(s);                      % Create Layout


%% Setting up a user track
% QuaDRiGa needs the positions of transmitter and receiver, e.g. for
% calculating the polarization or the arrival and departure angels. The
% positioning information of the Tx and Rx is essential also when the LSPs
% are not calculated. The following generates a linear track with 20 m
% length having a direction. The track is further split into 4 segments of
% 5 m length.
%
% The splitting is done by calling the method 'split_segment' of the track
% object. The the first two arguments of that function are the minimum
% length of a segment (1 m) and the maximum length of the segment (6 m).
% Each existing segment that is longer then the maximum length is split
% into subsegments. The length of those segments is random where the third
% and fourth parameter determine the mean and the STD of the length of new
% subsegment. Hence, 't.split_segment( 1,6,5,0 )' splits all segment longer
% than 6 m into subsegments of 5 m length.
%
% Each segment gets assigned a scenario. This is also essential since many
% parameters (such as the number of clusters, the XPR etc.) are
% scenario-specific. Hence, they are the same for the entire scenario.
% Here, we set the first the segments to NLOS, the third to LOS and the
% last to NLOS.
%
% Last, we set a random starting position for the track in the layout.

l.tx_position = [0;0;25];                       % Set Tx-position

t = track('linear',20);                         % Linear track, 20 m length
t.interpolate_positions( s.samples_per_meter);  % Interpolate to sample density
t.split_segment( 1,6,5,0 )                      % Split in 4 segments

Un = 'WINNER_UMa_C2_NLOS';
Ul = 'WINNER_UMa_C2_LOS';
t.scenario = {Un,Un,Ul,Un};                     % Set scenarios

l.randomize_rx_positions( 500,0,0,0 );          % Random start position
tmp = l.rx_position;
l.track = t;
l.rx_position = tmp;

l.visualize;

%% Manual setting of the parameters
% Now we initialize the parameter-set objects.
%
% The method "l.create_parameter_sets" splits the track into smaller
% sub-tracks, one for each segment. It further extracts the scenario
% informations. Each scenario gets its own parameter-set object. So we get
% an array of two parameter sets. The first element p(1) has three
% positions (NLOS segments) and the second has one position (LOS segment).

p = l.create_parameter_sets(1);

p(1).name
p(1).no_positions

p(2).name
p(2).no_positions

%%
% We set the number of clusters for the NLOS segments to 14. Currently, it
% is not possible to have a different number of clusters for each segment,
% i.e. it is not possible for the first NLOS segment to have 14 clusters
% and for the second to have only 10.

p(1).scenpar.NumClusters = 14;

%%
% In order to manually set the parameters, we have to overwrite the
% original settings. We do this here for the delay spread. The
% automatically generated values are:

[p(1).ds(1) p(1).ds(2) p(2).ds(1) p(1).ds(3) ]*1e6

%%
% We want to set the values of the four segments to 0.45, 0.33, 0.12 and
% 0.60 microseconds. This is done by:

p(1).ds(1) = 0.45e-6;
p(1).ds(2) = 0.33e-6;
p(2).ds(1) = 0.12e-6;
p(1).ds(3) = 0.60e-6;

%%
% The K-Factor and the shadow fading are read from the map during the
% generation of channel coefficients. This would overwrite any manual
% values. However, this could be deactivated. A drawback is that in this
% case the KF, SF and PL are only updated once per segment. This will result
% in a step-like function of the output power. There is currently no method
% the set the power manually on a per-snapshot basis.
%
% In the following example, we want to fix the received power to -102, -97,
% -82 and -99 dBm. K-Factors are taken from the map.

p(1).plpar = [];                        % Disable path loss for NLOS
p(2).plpar = [];                        % Disable path loss for LOS

p(1).sf = 10.^(0.1*[-102, -97, -99]);   % Set power for NLOS segments
p(2).sf = 10.^(0.1*[-82]);              % Set power for LOS segments

p(1).map_valid = false;                 % Disable automatic overwrite for NLOS
p(2).map_valid = false;                 % Disable automatic overwrite for LOS


%% Calculate channel coefficients
% Now we calculate the coefficients and the received power along the path.
% The following command calculate the channel coefficients. We then check
% the number of clusters that have been produced for each segment.

cm = p.get_channels;                    % Calculate the channel coefficients
cat( 1, cm.no_path )                    % Display the number of paths

%%
% The first three segments have 14 clusters. This has been set earlier. The
% last LOS segment has 15 clusters. One can see, that the three NLOS
% segment come first, followed by the LOS segment. The order has been
% scrambled. The following command sorts and combines the segments into one
% fading sequence.

c = cm.merge;

%%
% We now plot the power along the path. You can see the power levels of
% around -102, -97, -82 and -99 dBm which have been set earlier.
% Each segment has a transition area (e.g. from 2.5m to 5m, from 7.5m to
% 10m and from 12.5m to 15m) where the merging took place. In those areas,
% the power is scaled linearly. That means that, for example, in between 7.5
% and 10m, the power ramps up from -97 to -82 dBm.


power = squeeze(sum(abs(c.coeff).^2,3));
power = 10*log10(power);

figure
[~,dist] = t.get_length;
plot(dist,power)
title('Simulated Power')
xlabel('Distance from start point [m]')
ylabel('Received Power [dBm]')
axis([0,20,-110,-80])
grid on

%%
% The last plot shows the DS along the path. The results reflect the
% settings of 0.45, 0.33, 0.12 and 0.60 quiet well. As for the power, there
% is an overlap in between the segments. For example, in between 7.5 and
% 10m the DS drops from 0.33 to 0.12 microseconds. Additional fluctuations
% are caused by small scale fading.

coeff = squeeze( c.coeff );
delay = c.delay;

pow_tap = abs(coeff).^2;
pow_sum = sum(pow_tap);
mean_delay = sum( pow_tap.*delay) ./ pow_sum;
ds = sqrt( sum( pow_tap.*delay.^2)./ pow_sum - mean_delay.^2 );

figure
plot(dist,ds*1e6)
title('Simulated Delay Spread')
xlabel('Distance from start point [m]')
ylabel('RMS DS [\mus]')
axis([0,20,0,1])
grid on

%%
close all
disp(['QuaDRiGa Version: ',simulation_parameters.version])
