%% Effects of the Antenna-Orientation
%
% One feature of the model is that it allows to freely orient the antennas
% at the transmitter and receiver. In the following, two cross-polarized
% patch antennas are aligned on the optical axis facing each other. The
% surface normal vectors of the transmit and the receive patch are aligned
% with the LOS. The transmitter is rotated from -90° to 90° around the
% optical axis. The real and imaginary parts of the channel coefficients
% are then calculated for each angle. Each real and imaginary part is then
% normalized by its maximum and the results are plotted. The calculation is
% done for both, linearly and crossed polarized elements.

%% Model and Antenna Setup
% Here, we parametrize the simulation. We place the receiver 10 m away from
% the transmitter and chose the scenario "LOSonly". Thus, no NLOS
% components are present. The receiver is set up as a multi-element array
% using both, circular and linear polarization.

clear all
close all

s = simulation_parameters;
s.sample_density = 2;
s.show_progress_bars = 0;

l = layout(s);
l.track.generate('linear',0,0);
l.track.scenario = 'LOSonly';
l.rx_position = [11;0;0];
l.tx_position = [0;0;0];

a = array('rhcp-lhcp-dipole');
a.generate('custom',3,90,90,0);
a.set_grid( (-180:10:180)*pi/180 , (-90:10:90)*pi/180 );
a.rotate_pattern(180,'z',3);
a.copy_element(3,4);
a.rotate_pattern(90,'x',4);

b = a.copy_objects;
a.rotate_pattern(180,'z');

l.tx_array = a;
l.rx_array = b;

%% Initial parameter and coefficient generation

p = l.create_parameter_sets;                    % Create parameter sets
[~,cb] = p.get_channels;


%% Iteration over all angles
% Next, we iterate over all angles to calculate the channel response for
% different orientations of the tx antenna.  

rot = -120:10:120;
h = zeros(4,4,numel(rot));
for n=1:numel(rot)
    
    cc = a.copy_objects;
    cc.rotate_pattern( rot(n) , 'x');
    
    cb.tx_array = cc;
    c = cb.get_channels;
    
    h(:,:,n) = c.coeff(:,:,1,1);
end

%% Linear Polarization results

set(0,'defaultTextFontSize', 16)
set(0,'defaultAxesFontSize', 16)
set(0,'DefaultAxesFontName','Times')
set(0,'DefaultTextFontName','Times')

figure('Position',[ 100 , 100 , 660 , 400]);
g = h([3,4],[3,4],:);

plot(rot,squeeze(real(g(1,1,:))),'-sk','Linewidth',0.5,'MarkerfaceColor','k','Markersize',12)
hold on
plot(rot,squeeze(real(g(2,2,:))),'-db','Linewidth',0.5,'MarkerfaceColor','b','Markersize',8)
plot(rot,squeeze(real(g(2,1,:))),'-or','Linewidth',0.5,'MarkerfaceColor','r','Markersize',8)
plot(rot,squeeze(real(g(1,2,:))),'-^g','Linewidth',0.5,'MarkerfaceColor','g','Markersize',8)

plot(rot,squeeze(imag(g(1,1,:))),'--sk','Linewidth',0.5,'Markersize',12)
plot(rot,squeeze(imag(g(2,2,:))),'--db','Linewidth',0.5,'Markersize',8)
plot(rot,squeeze(imag(g(2,1,:))),'--or','Linewidth',0.5,'Markersize',12)
plot(rot,squeeze(imag(g(1,2,:))),'--^g','Linewidth',0.5,'Markersize',12)
hold off

xlabel('Rotaion Angle')
ylabel('Normalized Amplitude')



%% Circular Polarization results

figure('Position',[ 100 , 100 , 660 , 400]);
g = h([1,2],[1,2],:);

plot(rot,squeeze(real(g(1,1,:))),'-sk','Linewidth',0.5,'MarkerfaceColor','k','Markersize',12)
hold on
plot(rot,squeeze(real(g(2,2,:))),'-db','Linewidth',0.5,'MarkerfaceColor','b','Markersize',8)
plot(rot,squeeze(real(g(2,1,:))),'-or','Linewidth',0.5,'MarkerfaceColor','r','Markersize',8)
plot(rot,squeeze(real(g(1,2,:))),'-^g','Linewidth',0.5,'MarkerfaceColor','g','Markersize',8)

plot(rot,squeeze(imag(g(1,1,:))),'--sk','Linewidth',0.5,'Markersize',12)
plot(rot,squeeze(imag(g(2,2,:))),'--db','Linewidth',0.5,'Markersize',8)
plot(rot,squeeze(imag(g(2,1,:))),'--or','Linewidth',0.5,'Markersize',12)
plot(rot,squeeze(imag(g(1,2,:))),'--^g','Linewidth',0.5,'Markersize',12)
hold off

xlabel('Rotaion Angle')
ylabel('Normalized Amplitude')

%%
close all
disp(['QuaDRiGa Version: ',simulation_parameters.version])