function dist = interpolate_movement( obj , si , method )
%INTERPOLATE_MOVEMENT Interpolates the movement profile to a distance vector
%
%   dist = INTERPOLATE_MOVEMENT( si , method ) interpolates the movement
%   profile. The required input parameter 'si' is the sampling interval in
%   seconds. The optional parameter 'method' selects the interpolation
%   algorithm. The default is cubic. The output vector 'dist' is an ordered
%   list of positions on the track (distances from the start point) where
%   each value corresponds to one sampling point.     
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

if nargin == 3
supported_types = {'nearest','linear','spline','pchip','cubic'};
if ~( ischar(method) && any( strcmpi(method,supported_types)) )
    str = 'Array type not found; supported types are: ';
    no = numel(supported_types);
    for n = 1:no
        str = [str,supported_types{n}];
        if n<no
            str = [str,', '];
        end
    end
    error(str);
end
else
    method = 'pchip';
end

if ~( all(size(si) == [1 1]) && isnumeric(si) && isreal(si) && si >= 0 )
    error('??? Invalid sampling interval. The value must be real and > 0.')
end

if isempty(obj.movement_profile)
    obj.set_speed;
end

mp = obj.movement_profile;

max_time = mp(1,end);
max_dist = mp(2,end);

sr = si / ( 1-mod(max_time,si)/max_time );
t = 0 : sr : max_time ;
t(end) = max_time;

dist = interp1(mp(1,:),mp(2,:),t,method);

dist( dist<0 ) = 0;
dist( dist>max_dist ) = max_dist;


end

