function generate( obj, track_type , track_length , direction )
%GENERATE Generate simple (linear, circular) tracks
%
%   GENERATE( track_type ) creates a track of given type. Currently supported are
%   linear and circular tracks. If no further argument is given, the length of the
%   track will be 1 m. The direction of the linear track will be random. For the
%   circular track, the start point on the circle will be random.
%
%   GENERATE( track_type, track_length ) creates a track of given type and length.
%
%   GENERATE( track_type, track_length, direction ) creates a track of given type
%   and length. Direction is 2pi-periodic and describes the travel direction along
%   the track in mathematical sense (i.e. 0 means east, pi/2 means north, pi means
%   west and -pi/2 south). For the circular track, direction defines the starting
%   point on the circle. Positive values define the travel direction as counter
%   clock-wise and negative values as clock-wise. E.g. 0 sets the start point in
%   the east of the circle, traveling north; -2*pi sets it in the east, traveling
%   south.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

supported_types = {'circular','linear'};

% Check input arguemtns
if nargin == 1 || ~( ischar(track_type) && any( strcmpi(track_type,supported_types)) )
    str = 'Track type not found. Supported types are: ';
    no = numel(supported_types);
    for n = 1:no
        str = [str,supported_types{n}];
        if n<no
            str = [str,', '];
        end
    end
    error(str);
elseif nargin == 3
    direction = rand*2*pi;
elseif nargin == 2
    track_length = 1;
    direction = rand*2*pi;
end

if ~( isnumeric(track_length) &&...
        isreal(track_length) &&...
        all( size(track_length) == 1 ) &&...
        track_length >= 0 )
    error('??? "track_length" must be a real scalar > 0');
end

if ~( isnumeric(direction) &&...
        isreal(direction) &&...
        all( size(direction) == 1 ) )
    error('??? "direction" must be a real scalar');
end

% Generate track
switch track_type
    case 'circular'
        r = track_length/(2*pi);
        if direction >= 0
            pt = (0:1/128:1);
        else
            pt = (1:-1/128:0);
        end
        pt = pt + 0.5 * direction / pi;
        circ = r * exp( 2*pi*1j* pt );
        circ = circ - circ(1);
        obj.positions = [real(circ) ; imag(circ) ; zeros(1,numel(circ))];
        obj.positions( : ,obj.no_snapshots ) = obj.positions( : , 1 );
        obj.compute_directions;
        
    case 'linear'
        if track_length == 0
            obj.positions = [0;0;0];
            obj.ground_direction = angle( exp( 1j * direction ) );
        else
            r = 0.5 * track_length * exp( 1j * [ direction-pi ; direction ] );
            r = r-r(1);
            obj.positions = [ real(r(1)) , real(r(2)) ; imag(r(1)) , imag(r(2)) ; 0,0 ];
            obj.ground_direction = angle( exp( 1j * [direction,direction] ) );
        end
end

obj.segment_index = 1;
end

