function update_parameters( obj , force )
%UPDATE_PARAMETERS Updates the parameters for all user positions
%
%   UPDATE_PARAMETERS calculates correlated large scale parameters for each
%   user position. Those parameters are needed by the channel builder class
%   to calculate initial parameters for each track or segment which are
%   then evolved into time varying channels.
%
%   By default, "update_parameters" reads the values given in the track
%   objects of the layout. If there are no values given or if parts of the
%   values are missing, the correlation maps are generated to extract the
%   missing parameters.
%
%   The optional parameter "force=1" allows updatings the parameters even
%   when they are valid. This also discards all parameters specified in the
%   tracks.
%
%   For each user position, the uncorrelated LSPs are first taken from the
%   correlation maps. The second step introduces the cross-correlation by a
%   linear transformation of the seven parameters. The output is stored in
%   the class properties.
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input variables.
if nargin > 1
    force = real(force(1));
else
    force = 0;
end

% Update the correlations maps if neccesary
if force < 0
    % Do nothing
    % When "update_parameters" is called recursively for each
    % "parameter_set"-object, we do not need to check this again.
    
elseif force > 0
    obj.generate_correlation_maps;
    force = -1;         % Ignore values from track in any case
    
else % force == 0
    % Check if ALL parameters are provided by the tracks
    % (in this case, no maps are needed).
    
    par_complete = true;
    n=1; m=1;
    
    while par_complete && n <= numel(obj)
        
        % The loop stops when the first missing or incomplete set of
        % parameters is found. In this case, maps are generated.
        % When ALL parameters are complete (and the loop exits after
        % cheking all), then no maps will be generated.
        
        par = obj(n).track(m).par;
        if obj(n).no_positions == 0
            % There are no segments
            n=n+1;
            
        elseif isempty( par )
            % There are segments, but no parameters are given
            par_complete = false;
            
        elseif isempty( par.ds ) || isempty( par.kf ) || isempty( par.pg )  || ...
                isempty( par.asD ) || isempty( par.asA ) || isempty( par.esD ) || ...
                isempty( par.esA )
            % There are tracks with parameters, but the parameters are not
            % complete.
            
            par_complete = false;
        elseif m >= obj(n).no_positions
            % All segments of the current scenario have been checked
            n=n+1; m=1;
            
        else
            % There are unchecked segments in the current scenario.
            % Move to the next segment.
            m=m+1;
            
        end
    end
    
    % Calculate the maps only if there are parameters missing
    if ~par_complete
        % Determine which maps must be updated
        create_maps = ~cat( 2,obj.Pmap_valid ) & cat(2,obj.no_positions)>0;
        if any( create_maps )
            obj( create_maps ).generate_correlation_maps;
        end
        force = -2;     % Mixed values use maps for missing parameters
    else
        force = -3;     % Do not use maps and use only values from track
    end
    
end

if numel(obj) > 1
    % Do for parameter_set arrays
    for n=1:numel(obj)
        obj(n).update_parameters(force);
    end
    
elseif obj.no_positions > 0
    
    % Values of "force"
    %   -1    Ignore values from track in any case (force maps)
    %   -2    Mixed values use maps for missing parameters
    %   -3    Do not use maps and use only values from track (force track)
    
    if force ~= -3  % Interpolate values from the maps
        
        if any( min( obj.positions(1:2,:),[],2 ) ...
                < obj.map_extent(:, 1)) || ...
                any( max( obj.positions(1:2,:),[],2 ) ...
                > obj.map_extent(:, 2))
            error('Some positions are outside the map.');
        end

        % Interpolate the maps using a custom 2D spline interpolation
        ksi = spline_2d( obj.positions(1:2,:) , obj.parameter_maps ,...
            obj.map_x_coord , obj.map_y_coord );
        
        % Transform to linear values
        ksi( [1,4:7],: ) = 10.^( ksi( [1,4:7],: ) );
        ksi( [2,3],: )   = 10.^( 0.1 * ksi( [2,3],: ) );
        
    else
        ksi = zeros( 7,obj.no_positions );
    end
    
    % Read values from the tracks and overwrite data from the maps
    if force ~= -1
        par_fieldnames = {'ds','kf','pg','asD','asA','esD','esA'};
        
        for n = 1 : numel( obj.track )
            % Temporary copy of the par struct for faster access
            par = obj.track(n).par;
            
            if ~isempty( par )
                ind = obj.track(n).segment_index(end);
                
                for p = 1:7
                    % Temporaritly read values
                    tmp = par.( par_fieldnames{p} );
                    
                    % Copy the data to the parameter matrix
                    if ~isempty( tmp )
                        if p==2 || p==3
                            ksi( p,n ) = 10.^( 0.1 *tmp(1,ind) );
                        else
                            ksi( p,n ) = tmp(1,end);
                        end
                    end
                end
            end
        end
    end
    
    % Copy interpolated values to the output variables
    obj.ds  = ksi(1,:);
    obj.kf  = ksi(2,:);
    obj.sf  = ksi(3,:);
    obj.asD = ksi(4,:);
    obj.asA = ksi(5,:);
    obj.esD = ksi(6,:);
    obj.esA = ksi(7,:);
    
    obj.Pdata_valid = true;
end

end

% SUB FUNCTIONS
function i_x = x2i(x, x_range, samples_per_meter)
if (any(x > x_range(2))) || any((x < x_range(1)))
    error('x value has to be in x_range!');
end
i_x = floor((x-x_range(1))*samples_per_meter + 1);
end
