function [map , posx , posy ] = power_map( l , scen , usage , res ,...
    xs , ys , xe , ye   )
%POWER_MAP Calculates a power-map for the given layout
%
%    This functions calculates the average power seen by a terminal for a
%    given layout. This helps to predict the performance and for a given
%    setup. 
%
% The input variables are:
%    l       The Layout
%    scen    The scenario
%
%    usage   Sets the detail level
%       quick       Pattern, LOS Polarization and PL (default)
%       sf          Pattern, LOS Polarization, PL and SF
%       detailed    Full Simulation (one channel per pixel)
%
%    res     Distance of the sample points [m] (default = 10)
%
%    xs      x-coordinate of the top left corner
%    ys      y-coordinate of the top left corner
%    xe      x-coordinate of the bottom right corner
%    ye      y-coordinate of the bottom right corner
%
%
% Output variables:
%    map   a map for each tx-antenna
%          if multiple BS are defined, map is a cell array
%    posx  vector with the x-coordinates of the map
%    posy  vector with the y-coordinates of the map
%
% Example:
%    l = layout;
%    l.simpar.center_frequency = 2.68e9;
%
%    l.tx_position(:,1) = [0,100,20];
%    l.tx_array(1) = array('custom',20,20,0);
%
%    l.tx_position(:,2) = [0,100,45];
%    l.tx_array(2) = array('custom',20,20,0);
%    l.tx_array(2).rotate_pattern(-90,'z');
%
%    l.rx_array = array('dipole');
%
%    [map , posx , posy ] = power_map( l , 'LOSonly' , ...
%        -100 , 250 , 1000 , -420 , 20 );
%
%    figure(1)
%    imagesc(posx,posy,map{1});caxis([-5,25])
%    set(gca,'YDir','normal')
%
%    figure(2)
%    imagesc(posx,posy,map{2});caxis([-5,25])
%    set(gca,'YDir','normal')
%
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Default scenario
if ~exist( 'scen' , 'var' )
    scen = l.track(1).scenario{1};
end

% Set the usage mode
if exist( 'usage' , 'var' )
    if strcmp( usage , 'sf' ) ||...
            strcmp( usage , 'detailed' ) ||...
            strcmp( usage , 'quick' )
        % OK
    else
        error('Usage scenario not supported.')
    end
else
    usage = 'quick';
end

if ~exist( 'res' , 'var' )
    res = 10;
end

if nargin <= 4
    xs = min( l.tx_position(1,:)); 
    ys = max( l.tx_position(1,:));
    xe = max( l.tx_position(1,:));
    ye = min( l.tx_position(1,:));
    
    extend = max( [ 0.33*(xe-xs) , 0.33*(ys-ye) , 200 ] );
    
    xs = floor( (xs - extend)/res )*res;
    xe = ceil(  (xe + extend)/res )*res;
    ys = ceil(  (ys + extend)/res )*res;
    ye = floor( (ye - extend)/res )*res;
end


% Get the sample grid in x and y direction
posx = xs : res : xe;
posy = ys : -res : ye;

no_x = numel( posx );
no_y = numel( posy );
val  = no_x*no_y;

l.simpar.drifting_precision = 0;

% Create parameter sets for each tx-position
check = true;
for n = 1:l.no_tx
    if isa(scen,'parameter_set')
        q(n) = parameter_set;
        q(n).scenpar = scen.scenpar;
        q(n).plpar   = scen.plpar;
    else
        q(n) = parameter_set(scen,[],check);
        if n==1
            check = false;
        end
    end
    
    q(n).name = l.tx_name{n};
    
    q(n).simpar = l.simpar;
    
    q(n).tx_position = l.tx_position(:,n);
    q(n).no_positions = val;
    
    q(n).positions = [ repmat(posx,1,no_y) ;...
        reshape( repmat(posy,no_x,1) , 1 , [] ) ;...
        zeros(1,no_x*no_y) ];
    
    q(n).samples_per_meter = max( 2/res , 1/(3*pi) ); % 1/(3*pi)
    
    q(n).track = track('linear',0,0);
    q(n).rx_array = l.rx_array(1);
    
    q(n).tx_array = l.tx_array(n);
end

if strcmp( usage , 'sf' )
	% Calculate paraemter maps
    q.update_parameters;
    usage = 'quick';
end

switch usage
    case 'quick'
        for m = 1:l.no_tx
            c = channel_builder.get_los_channels( q(m));
            
            pow = permute( mean( abs(c.coeff).^2 , 1 ) , [4,2,1,3] );
            pow = reshape( pow , no_x , no_y , l.tx_array(m).no_elements );
            pow = permute(pow,[2,1,3]);
            map{m} = pow;
        end
        
    case 'detailed'
        % Create maps and parameters for each sample point
        q.update_parameters;
        
        % Calculate channels
        c = q.get_channels;
        
        % Calculate the maps
        for m = 1:l.no_tx
            ind = (m-1)*val;
            
            pow = zeros( val, l.tx_array(m).no_elements );
            for n = 1:val
                tmp = abs(c(n+ind).coeff(:,:,:,1)).^2;
                pow(n,:) = mean(sum(tmp,3),1);
            end
            pow = reshape( pow , no_x , no_y , l.tx_array(m).no_elements );
            pow = permute(pow,[2,1,3]);
            
            map{m} = pow;
        end
end

end

