function [ c,p,cb ] = get_channels( h_layout , sampling_rate , check_parfiles )
%GET_CHANNELS Calculate the channel coefficients
%
%   [ c,p,cb ] = GET_CHANNELS generates the channel coefficients.
%       c   is the channel object
%       p   is the parameter_set object
%       cb  is the channel_builder object
%
%   Options:
%       sampling_rate (default: 0.01 = 10 ms)
%       The sampling rate in seconds. This parameter is only use if a speed
%       profile is provided by the track object.
%
%       check_parfiles = 0 (default: 1)
%       Disables the parsing of shortnames and the validity-check for the
%       config-files. This is useful, if you know that the parameters in
%       the files are valid. In this case, this saves some execution time.
%
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input variables
if exist( 'sampling_rate','var' ) && ~isempty( sampling_rate )
    if ~( all(size(sampling_rate) == [1 1]) &&...
            isnumeric(sampling_rate) &&...
            isreal(sampling_rate) && sampling_rate > 0 )
        error('??? Invalid sampling interval. The value must be real and > 0.')
    end
else
    sampling_rate = 10e-3; % 10 ms
end

if ~exist( 'check_parfiles' , 'var' )
    check_parfiles = true;
end

verbose = h_layout.simpar.show_progress_bars;

% Create channel coefficients
p = h_layout.create_parameter_sets( 1,check_parfiles );
[c,cb] = p.get_channels;

if numel(c) > 1
    c = c.merge(0.5,1,verbose);
end

% Sort channel by name
n_channel = numel(c);
names = {};
for i_channel = 1:n_channel
    names{i_channel} = c(i_channel).name;
end
[names,ind] = sort(names);
c = c(ind);

% Determine, if chanel interpolation is needed
values = h_layout.no_rx;
tmp = { h_layout.track.movement_profile };
need_to_interpolate = false;
i_trk = 1;
while ~need_to_interpolate && i_trk <= values
    if ~isempty( tmp{i_trk} )
        need_to_interpolate = true;
    end
    i_trk = i_trk + 1;
end

if need_to_interpolate
    tStart = clock;
    if verbose; fprintf('Interpolate  ['); end; m0=0;
    
    % Apply speed profile, if provided
    channels_done = false(1,n_channel);
    for i_trk = 1 : h_layout.no_rx
        if verbose; m1=ceil(i_trk/values*50); if m1>m0; for m2=1:m1-m0; fprintf('o'); end; m0=m1; end; end;
        
        trk = h_layout.track(i_trk);
        
        if ~isempty( trk.movement_profile )
            dist = trk.interpolate_movement( sampling_rate );
            length = trk.get_length;
            
            for i_channel = 1 : n_channel
                if ~channels_done( i_channel )
                    if ~isempty( regexp( names{i_channel} ,  trk.name, 'once' ) )
                        c(i_channel) = c(i_channel).interpolate( dist , length , 'spline' );
                        channels_done( i_channel ) = true;
                    end
                end
            end
        end
    end
    
    if verbose
        fprintf('] %5.0f seconds\n',round( etime(clock, tStart) ));
    end
end

end
