function obj = generate(  varargin )
%GENERATE Generate predefined layout
%
%   GENERATE( 'regular' , no_sites , isd , ant )
%   generates a new multicell layout using a regular grid of BS positions.
%   Each BS has three sectors. The number of sites can be 1, 7 or 19 -
%   resulting in 3, 21 or 57 sectors, respectively. The 'isd' determines
%   the distance between the BS in m. The antenna 'ant' is the antenna for
%   one sector only. It will be rotated to match the sector orientations.
%   The broadside-direction of the provided antenna must be 0 (facing
%   east).
%
%   GENERATE( layout_type , layout_size , no_tx  )
%   generates a new layout with predefined settings.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

supported_types = {'random','regular'};

if nargin > 0
    layout_type = varargin{1};
end

% Check, if the layout type is supported
if nargin == 0 || ~( ischar(layout_type) && any( strcmpi(layout_type,supported_types)) )
    str = 'Layout type not found. Supported types are: ';
    no = numel(supported_types);
    for n = 1:no
        str = [str,supported_types{n}];
        if n<no
            str = [str,', '];
        end
    end
    error(str);
end


switch layout_type
    case 'regular'
        
        % Read the number of sites
        if nargin >= 2
            sites = varargin{2};
            if ~any(sites == [1,7,19])
                error('The number of sites must be 1,7, or 19');
            end
        else
            sites = 7;
        end
        
        % Read the ISD
        if nargin >= 3
            isd = varargin{3};
        else
            isd = 500;
        end
        
        % Read the antenna
        if nargin >= 4
            sector_ant = varargin{4};
            if ~isa(sector_ant,'array')
                error('The tx-antenna must be an array-class object.');
            end
        else
            sector_ant = array;
        end
        
        obj = layout;
        obj.no_tx = sites;
        
        % Set the positions of the BS sites
        if sites > 1
            tmp = 30+(0:5)*60;
            tmp = exp(1j*tmp*pi/180);
            obj.tx_position(1,2:7) = real( tmp );
            obj.tx_position(2,2:7) = imag( tmp );
        end
        if sites > 7
            tmp = 2*tmp;
            obj.tx_position(1,9:2:19) = real( tmp );
            obj.tx_position(2,9:2:19) = imag( tmp );
            dist = real(tmp(1));
            tmp = (0:5)*60;
            tmp = exp(1j*tmp*pi/180)*dist;
            obj.tx_position(1,8:2:18) = real( tmp );
            obj.tx_position(2,8:2:18) = imag( tmp );
        end
        obj.tx_position = obj.tx_position * isd;
        obj.tx_position(3,:) = 25;
        
        ant = sector_ant.copy;
        
        % Rotate antenna to match the sector orientation
        no_el = ant.no_elements;
        for n = 1 : no_el
            ant.copy_element( n , n + [1,2]*no_el );
        end
        ant.rotate_pattern( 30 , 'z' , 1:no_el );
        ant.rotate_pattern( 150 , 'z' , no_el+1 : 2*no_el );
        ant.rotate_pattern( -90 , 'z' , 2*no_el+1 : 3*no_el );
        
        obj.tx_array = ant;
        
        
    case 'random'
        
        if nargin >= 2
            layout_size = varargin{2};
        else
            layout_size = 500;
        end
        
        if nargin >= 3
            no_tx = varargin{3};
        else
            no_tx = 1;
        end
        
        
        if ~( isnumeric(layout_size) &&...
                isreal(layout_size) &&...
                all( size(layout_size) == 1 ) &&...
                layout_size > 0 )
            error('??? "layout_size" must be a real scalar > 0');
        end
        
        obj = layout;
        obj.no_tx = no_tx;
        
        tx_position_new = obj.tx_position;
        
        for n = 1:obj.no_tx
            a = rand*layout_size * exp( 2*pi*1j * rand );
            b = rand * (40 - 20) + 20;
            
            tx_position_new(1,n) = real(a);
            tx_position_new(2,n) = imag(a);
            tx_position_new(3,n) = b;
            
            obj.tx_position = tx_position_new;
        end
        
        obj.tx_array(1) = array('dipole');
        for n = 1:obj.no_tx
            obj.tx_array(n) = obj.tx_array(1);
        end
        
        obj.rx_array(1) = array('dipole');
        for n = 1:obj.no_rx
            obj.rx_array(n) = obj.rx_array(1);
        end
end

obj.name = layout_type;
end
