function [phi_d_1ms, theta_d_1ms, aod_nlos, eod_nlos, phi_a_lms,...
    theta_a_lms, psi_lms, tau_ls , lbs_pos ] ...
    = get_drifting(h_channel_builder, i_mobile)
%GET_DRIFTING Generate drifting angles and delays (private)
%
%   We assume that the positions of scatterers are fixed. As a consequence,
%   the scatter angles as seen from the transmitter (angles-of-departure,
%   AoDs) do not change. Based on the fixed-geometry assumption, however,
%   the scatter angles as seen from the MT (angles-of-arrival, AoAs) as
%   well as the sub-path delays along the track change due to the terminal
%   movement. This functions calls 'get_subpath_angles' to obtain initial
%   values for the departure- and arrival angles of all paths and subpaths.
%   Then, the drifting angles and delays are calculated geometrically. This
%   is done separately for the LOS (tx and rx) and NLOS paths (rx only).
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Read some common variables from the channel builder object. This
% increases speed since access to local variables is faster. 

n_clusters = h_channel_builder.par.scenpar.NumClusters;         % no. taps
n_snapshots = h_channel_builder.track(i_mobile).no_snapshots;   % no. snap
n_subpaths = 20;                                                % no. subpaths
lambda = h_channel_builder.simpar.wavelength;                   % wavelength
d_los = h_channel_builder.par.scenpar.LOS_scatter_radius;

drifting_precision = h_channel_builder.simpar.drifting_precision;
if drifting_precision >= 2
    % The travel direction for each snapshot
    gdir = h_channel_builder.track(i_mobile).ground_direction;
    
    % The element positions
    e_rx = h_channel_builder.rx_array(i_mobile).element_position;
    e_tx = h_channel_builder.tx_array.element_position;
    
    n_rx = h_channel_builder.rx_array(i_mobile).no_elements;
    n_tx = h_channel_builder.tx_array.no_elements;
else
    n_rx = 1;
    n_tx = 1;
    e_rx = [0;0;0];
    e_tx = [0;0;0];
end

% Create index_lists
% Again, this increases speed.
o_snap = ones(1,n_snapshots);
o_tx = ones(1,n_tx);
o_rx = ones(1,n_rx);
o_sub = ones(1,n_subpaths);
o_cluster = ones(1,n_clusters);

% We get the initial angles with random coupling
[phi_d_lm0, theta_d_lm0, phi_a_lm0, theta_a_lm0] =...
    h_channel_builder.get_subpath_angles(i_mobile);


% Reserve some memory for the output arguments
aod_nlos = phi_d_lm0(1, 2:n_clusters, :);       % AoD NLOS angles
eod_nlos = theta_d_lm0(1, 2:n_clusters, :);    	% EoD NLOS angles


% The distance vector from the Tx to the initial position
r_0 = h_channel_builder.track(i_mobile).initial_position - h_channel_builder.par.tx_position;
norm_r_0 = norm(r_0);


% Get the total path length of the NLOS component
d_l = h_channel_builder.taus(i_mobile, :).*h_channel_builder.simpar.speed_of_light + norm_r_0;


% Get the direction of the last bounce scatterer (LBS) seen from the
% receivers initial position.
[ahat_lm0_x, ahat_lm0_y, ahat_lm0_z] = sph2cart(phi_a_lm0, theta_a_lm0, 1);

% The distance of the LBS from the initial position
% We have a triangle where we know c=norm_r_0, a+b=d_l and cos(beta)=-r_0'*ahat_lm0/norm_r_0
% The cosine-theoreme: b^2 = a^2 + c^2 - 2*a*c*cos(beta)
% Substitute b=d_l-a and simplify (Note: Use -r_0 since we are at the
% receiver looking towards the transmitter)
r_0_T_ahat_lm0 = ...
    ahat_lm0_x .* repmat(r_0(1), [1, n_clusters, n_subpaths]) + ...
    ahat_lm0_y .* repmat(r_0(2), [1, n_clusters, n_subpaths]) + ...
    ahat_lm0_z .* repmat(r_0(3), [1, n_clusters, n_subpaths]);
norm_a_lm0 = 0.5 * repmat(d_l.^2 - norm_r_0.^2, [1, 1, n_subpaths]) ./...
    (repmat(d_l, [1, 1, n_subpaths]) + r_0_T_ahat_lm0);
norm_a_lm0( isnan( norm_a_lm0 ) ) = 0;


% The vector from the initial position to the position at snapshot s
q_s0 = h_channel_builder.track(i_mobile).positions;
if drifting_precision >= 2
    % We rotate the Rx array around the z-axis so it matches the travel
    % direction of the MT. Then we calculate the vector from the initial
    % position to each element of the Rx-array at snapshot s.

    % Note, "gdir" is a vector containing the travel direction for each
    % snapshot on the track.  
    
    % Temporal variables for the rotation
    c_gdir = cos(gdir);
    s_gdir = sin(gdir);
    
    e_rx_x = e_rx(1,:);
    e_rx_y = e_rx(2,:);
    e_rx_z = e_rx(3,:);

    % Apply the rotation
    q_s = zeros( 3,n_snapshots,n_rx );
    for i_rx = 1:n_rx
        q_s(1,:,i_rx) = c_gdir.*e_rx_x(i_rx) - s_gdir.*e_rx_y(i_rx);
        q_s(2,:,i_rx) = s_gdir.*e_rx_x(i_rx) + c_gdir.*e_rx_y(i_rx);
        q_s(3,:,i_rx) = e_rx_z(i_rx);
    end
    
    % Add the positions of the rotated elements to the Rx position on the
    % track at snapshot s.
    q_s = q_s + q_s0(:,:,o_rx);
else
    q_s = q_s0;
end


% The vector from the Rx position to the LBS
A = norm_a_lm0 .* ahat_lm0_x;
B = reshape( q_s(1,:,:) , n_snapshots , 1 , 1 , n_rx );
a_rlms_x = A( o_snap,:,:,o_rx ) - B( :,o_cluster,o_sub,: );

A = norm_a_lm0 .* ahat_lm0_y;
B = reshape( q_s(2,:,:) , n_snapshots , 1 , 1 , n_rx );
a_rlms_y = A( o_snap,:,:,o_rx ) - B( :,o_cluster,o_sub,: );

A = norm_a_lm0 .* ahat_lm0_z;
B = reshape( q_s(3,:,:) , n_snapshots , 1 , 1 , n_rx );
a_rlms_z = A( o_snap,:,:,o_rx ) - B( :,o_cluster,o_sub,: );

norm_a_rlms = sqrt(a_rlms_x.^2 + a_rlms_y.^2 + a_rlms_z.^2);


% The distance to the LOS scatterer will normally be 0. This will cause
% problems in the angle calculation
norm_a_rlms(norm_a_rlms==0) = 1e-6;


% The arrival angle
phi_a_lms = atan2(a_rlms_y, a_rlms_x);
theta_a_lms = asin(a_rlms_z./norm_a_rlms);


% We copy the >> arrival << angles for all Tx-Antennas.
% This assumes plane waves at the Tx >> only << for the departure angles. 
if drifting_precision >= 2
    phi_a_lms = phi_a_lms( :,:,:,:,o_tx );
    theta_a_lms = theta_a_lms( :,:,:,:,o_tx );
end


% The vector from the Tx to each Rx position on the track.
% r_s0 does not include the element positions.
r_s0_x = q_s0(1,:) + r_0(1);
r_s0_y = q_s0(2,:) + r_0(2);
r_s0_z = q_s0(3,:) + r_0(3);
norm_r_s0 = sqrt(r_s0_x.^2 + r_s0_y.^2 + r_s0_z.^2);

if drifting_precision >= 2
    % If we use high precision, we include the element positions in the
    % vector r_s. Hence, we get a vector for each MIMO sublink. 
    
    r_s_x = zeros( 1 , n_snapshots , n_rx , n_tx );
    r_s_y = zeros( 1 , n_snapshots , n_rx , n_tx );
    r_s_z = zeros( 1 , n_snapshots , n_rx , n_tx );
    for i_tx = 1:n_tx
        r_s_x(1,:,:,i_tx) = q_s(1,:,:) + r_0(1) - e_tx(1,i_tx);
        r_s_y(1,:,:,i_tx) = q_s(2,:,:) + r_0(2) - e_tx(2,i_tx);
        r_s_z(1,:,:,i_tx) = q_s(3,:,:) + r_0(3) - e_tx(3,i_tx);
    end
    norm_r_s = sqrt(r_s_x.^2 + r_s_y.^2 + r_s_z.^2);
       
    r_s_x = reshape( r_s_x , n_snapshots , 1,1,n_rx , n_tx );
    r_s_y = reshape( r_s_y , n_snapshots , 1,1,n_rx , n_tx );
    r_s_z = reshape( r_s_z , n_snapshots , 1,1,n_rx , n_tx );
    
    norm_r_s = reshape( norm_r_s , n_snapshots , 1,1,n_rx , n_tx );
else
    % With low precision, the vector r_s equals r_s0.
    
    r_s_x = r_s0_x.';
    r_s_y = r_s0_y.';
    r_s_z = r_s0_z.';
    norm_r_s = norm_r_s0.';
end


% Test if no position is smaller than d_los
if any( reshape( norm_r_s , [] , 1 ) <= d_los ) 
    error('get_drifting:norm_r_s', ['The distance between Tx and Rx has to be greater than %0.1f m.\n', ...
    'This is set via the LOS_scatter_radius parameter in the config file of the given scenario.'], d_los);
end


% The total distance
% Note: We assume a single-bounce cluster model here
if drifting_precision >= 2
    % Note, we calcualte b_lm0 from the values of r_s and a_lms at the
    % first snapshot position. The result is the same as when we use a_lm0
    % and r_rt0. However, we did not explicitely calculate a_lm0 and r_rt0. 
    % Hence, we save some computing time here.
    
    b_tlm0_x = r_s_x(1,o_cluster,o_sub,1,:) + a_rlms_x(1,:,:,1,o_tx);
    b_tlm0_y = r_s_y(1,o_cluster,o_sub,1,:) + a_rlms_y(1,:,:,1,o_tx);
    b_tlm0_z = r_s_z(1,o_cluster,o_sub,1,:) + a_rlms_z(1,:,:,1,o_tx);
    
    norm_b_tlm0 = sqrt(b_tlm0_x.^2 + b_tlm0_y.^2 + b_tlm0_z.^2);
    d_lms = norm_a_rlms(:,:,:,:,o_tx) + norm_b_tlm0(o_snap,:,:,o_rx,:);
else
    b_lm0 = d_l(1,:,o_sub) - norm_a_lm0;
    d_lms = norm_a_rlms + b_lm0( o_snap,:,: );
end


% The phases and delays
psi_lms = 2*pi/lambda * mod(d_lms, lambda);
tau_ls = mean(d_lms,3) ./ h_channel_builder.simpar.speed_of_light;


% For the LOS component, we assume that scattering is caused by local
% objects in close distance to the Tx or Rx. To simplify the simulation, we
% assume that all those scatterers have a constant distance 
% and are only separated by their angular distribution. Thus, they are
% placed on a circle. The variable 'd_LOS' is the radius of
% this circle.


% The departure LOS-angles at the Tx
A = atan2(r_s_y, r_s_x);
B = phi_d_lm0(1,1,:) - h_channel_builder.AoD(i_mobile, 1);
phi_d_1ms = A( :,1,o_sub,:,: ) + B( o_snap,1,:,o_rx,o_tx  );
   
A = asin(r_s_z./norm_r_s);
B = theta_d_lm0(1,1,:) - h_channel_builder.EoD(i_mobile, 1);
theta_d_1ms = A( :,1,o_sub,:,: ) + B( o_snap,1,:,o_rx,o_tx  );

% The arrival LOS-angles at the Rx
A = atan2(-r_s_y, -r_s_x);
B = phi_a_lm0(1,1,:) - h_channel_builder.AoA(i_mobile, 1);
phi_a_lms(:,1,:,:,:) = A( :,1,o_sub,:,: ) + B( o_snap,1,:,o_rx,o_tx  );

A = asin(-r_s_z./norm_r_s);
B = theta_a_lm0(1,1,:) - h_channel_builder.EoA(i_mobile, 1);
theta_a_lms(:,1,:,:,:) = A( :,1,o_sub,:,: ) + B( o_snap,1,:,o_rx,o_tx  );



% Calculate the phase of the LOS component

% Position of the LOS scatterer at the Tx and Rx
[b_x, b_y, b_z] = sph2cart(phi_d_lm0(1,1,:), theta_d_lm0(1,1,:), d_los);  % Tx
[a_x, a_y, a_z] = sph2cart(phi_a_lm0(1,1,:), theta_a_lm0(1,1,:), d_los);  % Rx

c_x = -b_x( o_snap,:,:,o_rx,o_tx  ) + r_s_x(:,1,o_sub,:,:) +...
    a_x( o_snap,:,:,o_rx,o_tx  );
c_y = -b_y( o_snap,:,:,o_rx,o_tx  ) + r_s_y(:,1,o_sub,:,:) +...
    a_y( o_snap,:,:,o_rx,o_tx  );
c_z = -b_z( o_snap,:,:,o_rx,o_tx  ) + r_s_z(:,1,o_sub,:,:) +...
    a_z( o_snap,:,:,o_rx,o_tx  );

d_1ms = 2*d_los + sqrt(c_x.^2 + c_y.^2 + c_z.^2);


if nargout == 9
    % Calculate the posiions of the LBS in global cartesian coordinates
    rx_pos = h_channel_builder.track(i_mobile).initial_position;
    
    lbs_pos = zeros( 3,n_clusters,20 );
    lbs_pos(1,:,:) = norm_a_lm0 .* ahat_lm0_x + rx_pos(1);
    lbs_pos(2,:,:) = norm_a_lm0 .* ahat_lm0_y + rx_pos(2);
    lbs_pos(3,:,:) = norm_a_lm0 .* ahat_lm0_z + rx_pos(3);
    
    lbs_pos(1,1,:) = lbs_pos(1,1,:) + a_x;
    lbs_pos(2,1,:) = lbs_pos(2,1,:) + a_y;
    lbs_pos(3,1,:) = lbs_pos(3,1,:) + a_z;
end


% Update phases and delays
psi_lms(:,1,:,:,:) = 2*pi/lambda * mod(d_1ms, lambda);
tau_ls(:,1,:,:,:) = norm_r_s ./ h_channel_builder.simpar.speed_of_light;
tau_ls = reshape( tau_ls , n_snapshots , n_clusters , n_rx , n_tx );


% When we use relative delays, we have to normalize the delays to the LOS

% tau_ls0 is the LOS delay at the RX-position without antennas. It is
% needed when the coefficients are going to be normalized to LOS delay.
if ~h_channel_builder.simpar.use_absolute_delays
    tau_ls0 = norm_r_s0.' ./ h_channel_builder.simpar.speed_of_light;
    tau_ls = tau_ls - tau_ls0(:,o_cluster,o_rx,o_tx);
end

end