function generate_xpr(obj)
%GENERATE_XPR Generate XPR (private)
%
% When a signal is reflected or scattered, its polarization axis changes.
% The XPR describes, how much power leaks from one polarization into the
% other. An XPR of 0 dB means that the power is spread equally, i.e. the
% polarization axis is turned by 45°. A value of -Inf means that the axis
% stays the same (and the signal keeps its polarization), a value of +Inf
% means that the polarization is turned by 90°. Other values are tabulated
% as follows
%
%    XPR      PV      PH    rotation
% -------------------------------------
%    -Inf      1       0       0
%    -6       7/8     1/8     20.7
%    -3       3/4     1/4     30
%     0       1/2     1/2     45
%    +3       1/4     3/4     60
%    +6       1/8     7/8     69.3
%    +Inf      0       1      90
%
% GENERATE_XPR generates a random (normal distributed) XPR value following
% having the distribution describes in scenapar (by mu and sigma). It then
% translates the XPR into a polarization rotation angle which is applied to
% the beam pattern.
%
% Note that all XPR values smaller that -40 are treated as -Inf and values
% greater +40 are treated as +Inf.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


L = obj.par.scenpar.NumClusters;                            % no. taps
N = obj.par.no_positions;                                   % no. positions

xpr_mu = obj.par.scenpar.xpr_mu;                            % XPR mean [dB]
xpr_sigma = obj.par.scenpar.xpr_sigma;                      % XPR std  [dB]

xpr = randn(N*L*20, 1) * xpr_sigma + xpr_mu;

if obj.simpar.use_polarization_rotation
    
    % Conversion of the xpr into one rotation angle
    xpr = acot(sqrt(10.^(0.1*xpr)) );
    
    % We assume no XPR for the LOS component
    obj.xpr = reshape(xpr, N, L, 20);
    obj.xpr(:, 1, :) = 0;
    
else
    
    obj.xpr = reshape(10.^(xpr/10), N, L, 20);
    obj.xpr(:, 1, :) = 0;
    
end

end
