function generate_initial_paths(obj)
%GENERATE_INITIAL_PATHS Generate delays and powers (private)
%
% The generation of initial delays is done according to the winner WIM2
% model. The only difference is that here we assume that the LOS component
% is always present. In a heavily shaded environment, however, the K-factor
% becomes very small (below 0). Delays are drawn randomly from a delay
% distribution defined in the parameter set. This distribution can be
% estimated from measurements and is scenario dependent.
%
% Since a delay represents a longer path-distance, the signal will also
% have a higher attenuation. The NLOS cluster powers are calculated
% assuming a single slope exponential power delay profile. Power assignment
% depends on the delay spread and the per cluster shadow fading which is
% predefined in the parameter database.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


L = obj.par.scenpar.NumClusters;                            % no. taps
N = obj.par.no_positions;                                   % no. positions

ds = obj.par.ds.';

if L ~= 1                                                   % LOS and NLOS components
    
    % First, we create the path delays
    taus = ds(:, ones(1, L));                               % initialize output with RMS-DS
    taus = -obj.par.scenpar.r_DS * taus .* log(rand(N, L)); % generate random log-normal varaibles
    taus = sort(taus, 2);                                  	% sort output
    taus = taus - taus(:, ones(1, L));                      % normalize min. delay to zero
    
    % Generate the cluster powers
    % Note that the K-Factor correction is not yet apllied to the delays.
    
    pow = exp(-taus .*...
        (obj.par.scenpar.r_DS - 1)  ./...
        (obj.par.scenpar.r_DS * ds(:, ones(1, L)))  );
    
    % Apply per-path shadowing
    pow = pow .* 10.^(-randn(N, L) * obj.par.scenpar.LNS_ksi * 0.1);
    obj.pow_wo_kf = pow;
    
    % Apply K-Factor in formation to cluster powers.
    pow(:, 1) = obj.par.kf' .* sum(pow(:, 2:L), 2);
    
    % Normalize the path power such that the sum is equal to 1
    for n = 1 : N
        pow(n, :) = pow(n, :) / sum(pow(n, :));
    end
    
    % Normalize the taus to match the given delay spread
    ds_out = abs(sqrt(sum(pow.*(taus.^2), 2)  - sum(pow.*taus, 2).^2)  );
    scale = ds./ds_out;
    taus = scale(:, ones(1, L)).*taus;
    
else
    pow = ones(N, 1);
    taus = zeros(N, 1);
end

obj.taus = taus;
obj.pow = pow;

end
