function Cc = correction_function(L, K)
%CORRECTION_FUNCTION Corrects the initial RMS Angular spread
%
% The correction function C takes the influence of the K-Factor and the
% varying number of clusters on the angular spreads into account. To
% approximate the function, we generated the angles as implemented in
% 'generate_initial_angles' with C set to one. Then we calculated the
% angular spread from the simulated data and compare the output of the
% procedure with the given value.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

N = numel(K);
if numel(L) == 1
    L = ones(1, N)*L;
end

% Set maximum range
L(L<3) = 3;
L(L>42) = 42;
K(K<-20) = -20;
K(K>20) = 20;

% The correction function at some initial values
cL = [3, 6, 9, 12, 15, 18, 21, 24, 27, 30, 33, 36, 39, 42];
cK = [-20, -18, -16, -14, -12, -10, -8, -6, -4, -2, 0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20];
C = [3.59, 5.53, 6.25, 6.62, 6.81, 6.93, 7.04, 7.11, 7.17, 7.13, 7.22, 7.22, 7.25, 7.31; ...
    3.76, 5.63, 6.34, 6.71, 6.78, 6.98, 7.05, 7.07, 7.19, 7.18, 7.23, 7.21, 7.28, 7.26; ...
    4.00, 5.76, 6.36, 6.74, 6.88, 6.96, 7.08, 7.09, 7.18, 7.14, 7.18, 7.24, 7.26, 7.28; ...
    4.24, 5.90, 6.51, 6.78, 6.89, 6.92, 6.99, 7.05, 7.08, 7.16, 7.14, 7.15, 7.06, 7.10; ...
    4.54, 5.98, 6.47, 6.68, 6.74, 6.84, 6.86, 6.92, 6.86, 7.03, 7.38, 7.80, 8.36, 8.77; ...
    4.74, 5.98, 6.38, 6.51, 6.55, 6.66, 7.07, 7.76, 8.54, 9.03, 9.65, 10.12, 10.55, 10.97; ...
    4.83, 5.80, 5.99, 6.27, 7.27, 8.38, 9.12, 9.89, 10.45, 10.90, 11.44, 11.74, 12.10, 12.35; ...
    4.70, 5.30, 6.42, 7.96, 9.13, 10.01, 10.69, 11.36, 11.78, 12.15, 12.60, 13.01, 13.15, 13.47; ...
    4.12, 5.72, 8.00, 9.42, 10.34, 11.09, 11.52, 12.16, 12.57, 12.86, 13.20, 13.39, 13.72, 14.02; ...
    3.42, 6.94, 8.88, 10.01, 10.71, 11.32, 11.90, 12.15, 12.65, 12.95, 13.17, 13.37, 13.53, 13.75; ...
    4.19, 7.69, 9.07, 10.05, 10.68, 11.09, 11.58, 11.76, 12.12, 12.36, 12.48, 12.84, 12.95, 13.03; ...
    5.09, 7.66, 8.76, 9.53, 9.98, 10.42, 10.68, 10.99, 11.20, 11.31, 11.45, 11.62, 11.79, 11.87; ...
    5.33, 7.22, 8.17, 8.67, 9.07, 9.33, 9.53, 9.81, 9.93, 10.01, 10.24, 10.31, 10.44, 10.59; ...
    5.21, 6.64, 7.26, 7.63, 7.94, 8.10, 8.33, 8.51, 8.62, 8.63, 8.82, 8.89, 9.00, 9.08; ...
    4.80, 5.85, 6.30, 6.60, 6.83, 6.97, 7.12, 7.23, 7.36, 7.38, 7.43, 7.53, 7.62, 7.67; ...
    4.29, 5.02, 5.38, 5.59, 5.75, 5.87, 5.97, 6.05, 6.11, 6.21, 6.26, 6.30, 6.33, 6.37; ...
    3.75, 4.28, 4.54, 4.68, 4.79, 4.90, 4.96, 5.01, 5.10, 5.13, 5.19, 5.22, 5.24, 5.27; ...
    3.20, 3.59, 3.77, 3.90, 3.97, 4.06, 4.11, 4.17, 4.20, 4.23, 4.26, 4.28, 4.32, 4.35; ...
    2.70, 2.98, 3.11, 3.21, 3.28, 3.33, 3.37, 3.41, 3.44, 3.46, 3.49, 3.52, 3.54, 3.56; ...
    2.25, 2.46, 2.56, 2.63, 2.68, 2.72, 2.76, 2.78, 2.80, 2.83, 2.85, 2.86, 2.88, 2.89; ...
    1.86, 2.01, 2.10, 2.14, 2.18, 2.22, 2.24, 2.26, 2.27, 2.29, 2.31, 2.32, 2.33, 2.34];

noL = numel(cL);
noK = numel(cK);

% The linear interpolation for intermediate values
[tmp, b] = sort(L);
[~, a] = sort([cL, tmp]);
ui = 1:(noL + N);
ui(a) = ui;
ui = ui(noL+1:end) - (1:N);
ui(b) = ui;
ui(ui == noL) = noL-1;
uin = ui+1;
u = (L-cL(ui))./(cL(uin)-cL(ui));
u = u';

[tmp, b] = sort(K);
[~, a] = sort([cK, tmp]);
vi = 1:(noK + N);
vi(a) = vi;
vi = vi(noK+1:end) - (1:N);
vi(b) = vi;
vi(vi == noK) = noK-1;
vin = vi+1;
v = (K-cK(vi))./(cK(vin)-cK(vi));
v = v';

c1 = (1-v).*(1-u);
c2 = (1-v).*u;
c3 = v.*(1-u);
c4 = v.*u;

pa = vi  + (ui  -1)*noK;
pb = vi  + (uin -1)*noK;
pc = vin + (ui  -1)*noK;
pd = vin + (uin -1)*noK;

ndx = [pa, pb, pc, pd];
a = C(ndx);
a = reshape(a, N, 4);

% The output
Cc = c1.*a(:, 1) + c2.*a(:, 2) + c3.*a(:, 3) + c4.*a(:, 4);
end
