function h_channel = get_channels( h_channel_builder , vb_dots )
%GET_CHANNELS Generates channel coefficients
%
%   c = GET_CHANNELS generates the channel coefficients. This is
%   the main function of the channel builder.
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

verbose = h_channel_builder(1).simpar.show_progress_bars;
if verbose && nargin == 1
    fprintf('Channels     [');
    vb_dots = 50;
    tStart = clock;
end
m0=0;

if numel(h_channel_builder) > 1
    vb_dots = zeros( 1,numel(h_channel_builder) );
    if verbose
        for n_channel_builders = 1:numel(h_channel_builder)
            vb_dots(n_channel_builders) = ...
                h_channel_builder(n_channel_builders).par.no_positions;
        end
        vb_dots = round( vb_dots/sum(vb_dots) * 50 );
    end
    [~,tmp] = max(vb_dots);
    vb_dots(tmp) = vb_dots(tmp)+50-sum(vb_dots);
    
    h_channel = channel.empty;
    for n_channel_builders=1:numel(h_channel_builder)
        tmp = h_channel_builder(n_channel_builders).get_channels(vb_dots(n_channel_builders));
        h_channel = [ h_channel , tmp ];
    end
    
else
    % These variables are often needed. Pre-computing them saves a lot of time
    use_subpath_output = h_channel_builder.simpar.use_subpath_output;
    use_polarization_rotation = h_channel_builder.simpar.use_polarization_rotation;
    map_valid = h_channel_builder.par.map_valid;
    drifting_th = h_channel_builder.simpar.drifting_update_threshold * pi/180;
    wave_no = 2*pi/h_channel_builder.simpar.wavelength;
    initial_rx_position = h_channel_builder.par.positions;
    drifting_precision = h_channel_builder.simpar.drifting_precision;


    % Access to class-properties is time consuming.
    % The array interpolation is the most time intense operation in the
    % channel builder. We save some computing time by reading the arrays
    % here and passing them as varaibles to the interpolate function later
    % on.
    tx_elevation_grid = h_channel_builder.tx_array.elevation_grid;
    tx_azimuth_grid = h_channel_builder.tx_array.azimuth_grid;
    tx_patV = h_channel_builder.tx_array.field_pattern_vertical;
    tx_patH = h_channel_builder.tx_array.field_pattern_horizontal;
    tx_CP = exp( 1j * h_channel_builder.tx_array.common_phase );
    tx_element_pos = h_channel_builder.tx_array.element_position;
    
    % Just for easier handling and faster data access
    n_clusters  = h_channel_builder.par.scenpar.NumClusters;	% no. taps
    n_subpaths = 20;                                            % no. subpaths
    n_paths = n_clusters*n_subpaths;                           	% no. taps and subpaths
    n_mobiles  = h_channel_builder.par.no_positions;           	% no. positions
    
    o_sub = ones(1,n_subpaths);                                 % subpath index list
    o_clusters = ones(1,n_clusters);
    
    % Generate the initial parameters here
    % If the parameters are already given in the channel builder object,
    % we use the given parameters.
    
    if isempty(h_channel_builder.taus)
        generate_initial_paths( h_channel_builder );
    end
    
    if isempty(h_channel_builder.AoD)
        generate_initial_angles( h_channel_builder );
    end
    
    if isempty(h_channel_builder.xpr)
        xpr_mu = h_channel_builder.par.scenpar.xpr_mu;          % XPR mean [dB]
        xpr_sigma = h_channel_builder.par.scenpar.xpr_sigma;   	% XPR std  [dB]
        h_channel_builder.xpr = randn(n_mobiles,n_clusters,n_subpaths) * xpr_sigma + xpr_mu;
        h_channel_builder.xpr(:,1,:) = Inf;
    end
    
    if isempty(h_channel_builder.pin)       % Random initial phases
        h_channel_builder.pin = ( rand(n_mobiles,n_paths)-0.5 )*2*pi;
    end
    
    if isempty(h_channel_builder.kappa)     % Random polarization phases
        h_channel_builder.kappa = (rand(4,n_paths,n_mobiles)*2*pi - pi);
    end
    
    if isempty(h_channel_builder.subpath_coupling)
        [~,h_channel_builder.subpath_coupling] =...
            sort(rand(n_subpaths,4,n_clusters,n_mobiles));
    end
    
    
    if drifting_precision == 0 || map_valid == 0
        % Get the path loss for each rx position.
        % When drifting is enabled, the path loss is also drifting und will
        % be calculated separately.
        [ path_loss , scale_sf ] = h_channel_builder.par.get_pl;
        rx_power = -path_loss.' + 10*log10( h_channel_builder.par.sf ) .* scale_sf;
        rx_power = sqrt( 10.^( 0.1 * rx_power ) );
    end
    
    % The loop for each user position
    h_channel = channel.empty(n_mobiles,0);
    for i_mobile = 1 : n_mobiles
        if verbose; m1=ceil(i_mobile/n_mobiles*vb_dots); if m1>m0;
                for m2=1:m1-m0; fprintf('o'); end; m0=m1; end;
        end;
        
        % It is possible for all usrers to share the same track object. In
        % this case, the initial position will be the same for all tracks.
        % However, in the parameter_set object, different initial positions
        % can be specified. The following statement compares both values
        % and corrects the initial position in the tack object, if needed.
        
        if any( h_channel_builder.track(i_mobile).initial_position ~= ...
                initial_rx_position(:,i_mobile) )
            h_channel_builder.track(i_mobile).initial_position =...
                initial_rx_position(:,i_mobile);
        end
        
        % Read some commonly needed variables in order to save time.
        n_rx        = h_channel_builder.rx_array(i_mobile).no_elements;
        n_tx        = h_channel_builder.tx_array.no_elements;
        n_links     = n_rx*n_tx;
        n_snapshots = h_channel_builder.track(i_mobile).no_snapshots;
        initial_pos = h_channel_builder.track(i_mobile).segment_index( ...
            min( [h_channel_builder.track(i_mobile).no_segments,2] ));
        
        % Access to class-properties is time consuming.
        % The array interpolation is the most time intense operation in the
        % channel builder. We save some computing time by reading the arrays
        % here and passing them as varaibles to the interpolate function.
        rx_elevation_grid = h_channel_builder.rx_array(i_mobile).elevation_grid;
        rx_azimuth_grid = h_channel_builder.rx_array(i_mobile).azimuth_grid;
        rx_patV = h_channel_builder.rx_array(i_mobile).field_pattern_vertical;
        rx_patH = h_channel_builder.rx_array(i_mobile).field_pattern_horizontal;
        rx_CP = exp( 1j * h_channel_builder.rx_array(i_mobile).common_phase );
        rx_element_pos = h_channel_builder.rx_array(i_mobile).element_position;
        
        % Create arrays of ones to speed up indexing
        o_tx      = ones(1,n_tx);
        o_rx      = ones(1,n_rx);
        
        % Extract the random initial phases
        pin = h_channel_builder.pin(i_mobile,:);
        
        % We need the directions. If they are not provided, compute them here.
        if isempty(h_channel_builder.track(i_mobile).ground_direction)
            h_channel_builder.track(i_mobile).compute_directions;
        end
        
        switch drifting_precision
            case 0
                % If we don't use drifting and have a linear track, then the
                % Doppler component is only dependent on the rotating phases of the
                % taps. So, we don't recalculate the antenna response for each
                % snapshot.
                
                % Get the angles of the 20 subpaths and perform random coupling.
                [ aod,eod,aoa,eoa,delay ] =...
                    h_channel_builder.get_subpath_angles(i_mobile);
                
                % Doppler component
                % Without drifting, the Doppler component is calculated by
                % plane wave approximation using the distance from the initial
                % position.
                tmp = h_channel_builder.track(i_mobile).positions;
                dist = sqrt( sum([ tmp(1,:) - tmp(1,1) ; ...
                    tmp(2,:) - tmp(2,1)   ; ...
                    tmp(3,:) - tmp(3,1)   ].^2 ) );
                
                % Generate the tx-array channel coefficients for each user position.
                % Since the receiver is mobile, we have to adjust for the movement direction
                % inside the loop.
                
                [Vt,Ht,CPt,Pt] = h_channel_builder.tx_array.interpolate( ...
                    aod, eod, 1:n_tx, tx_azimuth_grid, tx_elevation_grid, ...
                    tx_patV, tx_patH, tx_CP, tx_element_pos );
                Pt = reshape( Pt, 1 ,n_paths, n_tx );
                CPt = reshape( CPt, 1, n_paths, n_tx );
                no_snap_process = 1;
                
            case 1
                % Get the drifting angles
                [ aod_los,eod_los , aod_nlos,eod_nlos , aoa,eoa , phase,delay ] ...
                    = h_channel_builder.get_drifting(i_mobile);
                no_snap_process = n_snapshots;
                
                phase = reshape( phase , n_snapshots , n_paths );
                aod = [ aod_los(1,1,:) aod_nlos ];
                eod = [ eod_los(1,1,:) eod_nlos ];
                
                % Precalculate the Tx-Array-Pattern response for the first snapshot
                [Vt,Ht,CPt,Pt] = h_channel_builder.tx_array.interpolate(...
                    aod, eod, 1:n_tx, tx_azimuth_grid, tx_elevation_grid,...
                    tx_patV, tx_patH, tx_CP, tx_element_pos );
                Pt = reshape( Pt, 1, n_paths, n_tx );
                CPt = reshape( CPt, 1, n_paths, n_tx );
                
            otherwise  % case 2 and 3
                % Get the drifting angles
                [ aod_los,eod_los , aod_nlos,eod_nlos , aoa,eoa , phase,delay ] ...
                    = h_channel_builder.get_drifting(i_mobile);
                no_snap_process = n_snapshots;
                
                phase = reshape( phase , n_snapshots , n_paths , n_rx , n_tx );
                aod = [ aod_los(1,1,:,:,:) aod_nlos(:,:,:,o_rx,o_tx) ];
                eod = [ eod_los(1,1,:,:,:) eod_nlos(:,:,:,o_rx,o_tx) ];
                
                % Precalculate the Tx-Array-Pattern response for each
                % element separately.
                Vt = zeros(1,n_clusters,n_subpaths,n_rx,n_tx);
                Ht = zeros(1,n_clusters,n_subpaths,n_rx,n_tx);
                CPt = zeros(1,n_clusters,n_subpaths,n_rx,n_tx);
                for i_tx = 1:n_tx
                    [Vt(:,:,:,:,i_tx),Ht(:,:,:,:,i_tx),CPt(:,:,:,:,i_tx)] =...
                        h_channel_builder.tx_array.interpolate( ...
                        aod(:,:,:,:,i_tx), eod(:,:,:,:,i_tx), i_tx, ...
                        tx_azimuth_grid, tx_elevation_grid, tx_patV,...
                        tx_patH, tx_CP, tx_element_pos );
                end
                CPt = reshape( CPt, 1, n_paths, n_rx, n_tx );
        end
        
        % Travel directions
        gdir = h_channel_builder.track(i_mobile).ground_direction;
        
        % XPR-Values and array indices
        if use_polarization_rotation
            
            % Conversion of the XPR into one rotation angle
            xpr = acot( sqrt( 10.^(0.1*h_channel_builder.xpr(i_mobile,:,:)) ) );
            
            % Here we initialize the variables for the polarization
            % rotation method.
            
            xpr = reshape( xpr,1,1,n_clusters,n_subpaths );             % Rotation angles
            xprmat = zeros(4,n_paths);                                  % Initialization
            PolR = h_channel_builder.rx_array(i_mobile).pol_vector;     % Pol-Vectors
            
            kappa = exp(1j*h_channel_builder.kappa(1,:,i_mobile));      % Optional HV phase offset
            kappa( 1:n_clusters:n_paths ) = 1;                          % No elliptical Pol for LOS
            
        else
            
            % This is the polarization coupling from WINNER. The
            % polarization is initialized by random phases which are scaled
            % by the XPR. Polarization drifting is not supported.
            
            % XPR-Values
            xpr =  10.^(h_channel_builder.xpr(i_mobile,:,:)/10);
            xpr = sqrt( reshape( 1./xpr ,1,n_paths ) );
            
            % Random initial phases
            xprmat = exp( 1j*h_channel_builder.kappa(:,:,i_mobile));
            
            % Global XPR-Matrix
            xprmat = xprmat .* [ones(1,n_paths);xpr;xpr;ones(1,n_paths)];
            
            % Identitiy Matrix for the LOS-Path
            xprmat( 1,1:n_clusters:n_paths ) = 1;
            xprmat( 2,1:n_clusters:n_paths ) = 0;
            xprmat( 3,1:n_clusters:n_paths ) = 0;
            xprmat( 4,1:n_clusters:n_paths ) = 1;
        end
        
        c  = zeros( n_links , n_paths , n_snapshots );          % The pattern coeficient matrix
        cp = zeros( n_links , n_paths , n_snapshots );          % The phase coeficient matrix
        pow_pat = zeros( n_links , n_paths , no_snap_process ); % Store the radiated power of the patterns
        
        for i_snapshot = 1 : no_snap_process                     % Track positions
            
            % Include the direction on travel in the angles of arrival
            % This rotates the terminal around the z-axis by turning the
            % pattern. Rotations around z do not effect the polarization.
            aoa_c = aoa(i_snapshot,:,:,:,:) - gdir(i_snapshot);
            eoa_c = eoa(i_snapshot,:,:,:,:);
            
            % Calculate the receiver array response for the current snapshot
            if drifting_precision >= 2
                Vr = zeros(1,n_clusters,n_subpaths,n_rx,n_tx);
                Hr = zeros(1,n_clusters,n_subpaths,n_rx,n_tx);
                CPr = zeros(1,n_clusters,n_subpaths,n_rx,n_tx);
                for i_rx = 1:n_rx
                    [Vr(:,:,:,i_rx,:),Hr(:,:,:,i_rx,:),CPr(:,:,:,i_rx,:)] =...
                        h_channel_builder.rx_array(i_mobile).interpolate( ...
                        aoa_c(:,:,:,i_rx,:) , eoa_c(:,:,:,i_rx,:) , i_rx , ...
                        rx_azimuth_grid , rx_elevation_grid , rx_patV , ...
                        rx_patH, rx_CP, rx_element_pos);
                end
                CPr = reshape( CPr, 1, n_paths, n_rx, n_tx );
                
            else
                [Vr,Hr,CPr,Pr] = h_channel_builder.rx_array(i_mobile).interpolate( aoa_c ,...
                    eoa_c , 1:n_rx , rx_azimuth_grid , rx_elevation_grid ,...
                    rx_patV, rx_patH, rx_CP, rx_element_pos );
                Prn = reshape( Pr,1,n_paths,n_rx );
                CPr = reshape( CPr,1,n_paths,n_rx );
            end
            
            % When drifting is enabled, we have to update the transmitter
            % response for the LOS path.
            if i_snapshot > 1
                if drifting_precision >= 2
                    tmp = zeros( 1, 1, n_subpaths, n_rx, n_tx );
                    for i_tx = 1:n_tx
                        [Vt(1,1,:,:,i_tx), Ht(1,1,:,:,i_tx), tmp(1,1,:,:,i_tx)] =...
                            h_channel_builder.tx_array.interpolate( ...
                            aod_los(i_snapshot,1,:,:,i_tx) , eod_los(i_snapshot,1,:,:,i_tx) ,...
                            i_tx, tx_azimuth_grid, tx_elevation_grid ,...
                            tx_patV, tx_patH, tx_CP, tx_element_pos );
                    end
                    CPt(1,1:n_clusters:n_paths,:,:) = reshape( tmp, 1, n_subpaths, n_rx, n_tx );
                    
                else
                    [Vt(1,1,:,:),Ht(1,1,:,:),CPt_tmp,tmp] =...
                        h_channel_builder.tx_array.interpolate( ...
                        aod_los(i_snapshot,1,:) , eod_los(i_snapshot,1,:) , 1:n_tx , ...
                        tx_azimuth_grid, tx_elevation_grid, tx_patV, tx_patH, tx_CP, tx_element_pos );
                    Pt(1,1:n_clusters:n_paths,:)  = reshape( tmp, 1, n_subpaths, n_tx );
                    CPt(1,1:n_clusters:n_paths,:) = reshape( CPt_tmp, 1, n_subpaths, n_tx );
                end
            end
            
            if use_polarization_rotation
                % Drifting of the polarization is very computing intensive.
                % Thus, we only update the polarization when the arrival angles
                % change more than a given threshold which is defined in the
                % simulation parameters. This reduces the running time by up to
                % a factor of 10.
                
                if i_snapshot == 1
                    deg = zeros( n_rx , n_tx , n_clusters , n_subpaths );
                    last_update = ones(1,n_clusters);
                    update_paths = true(1,n_clusters);
                else
                    update_paths = false( 1,n_clusters );
                    
                    aoa_change = abs( diag( aoa( last_update , : , 1) ).' - ...
                        aoa( i_snapshot,:,1) );
                    
                    eoa_change = abs( diag( eoa( last_update , : , 1) ).' - ...
                        eoa( i_snapshot,:,1 ) );
                    
                    gdir_change = abs( gdir( last_update ) - gdir(i_snapshot) );
                    
                    tmp = aoa_change > drifting_th |...
                        eoa_change > drifting_th |...
                        gdir_change > drifting_th;
                    
                    update_paths( tmp ) = true;
                    last_update( tmp ) = i_snapshot;
                end
                
                % Apply the orientation of the Rx-antenna to the orientation
                % vector.
                tmp = gdir(i_snapshot);
                Rz = makehgtform('zrotate', tmp);
                Qr = Rz(1:3, 1:3) * PolR;
                
                i_cluster = find(update_paths);
                if drifting_precision >= 2
                    % With high precision, we need to calculate the
                    % polarization rotation for each MIMO-Link
                    % separately.
                    deg(:,:,i_cluster,:) = calc_polarization_rotation( ...
                        aoa( i_snapshot,i_cluster,:,:,: ),...
                        eoa( i_snapshot,i_cluster,:,:,: ),...
                        Qr,...
                        Vr(1,i_cluster,:,:,:),...
                        Hr(1,i_cluster,:,:,:),...
                        xpr(1,1,i_cluster,:) );
                    
                else
                    % With low precision, we can calculate the
                    % polarization rotation for all antennas
                    % together (much faster).
                    deg(:,1,i_cluster,:) = calc_polarization_rotation( ...
                        aoa( i_snapshot,i_cluster,: ),...
                        eoa( i_snapshot,i_cluster,: ),...
                        Qr,...
                        Vr(1,i_cluster,:,:),...
                        Hr(1,i_cluster,:,:),...
                        xpr(1,1,i_cluster,:) );
                    deg(:,:,i_cluster,:) = deg(:,o_tx,i_cluster,:);
                end
                degu = reshape( deg , n_rx*n_tx , n_paths );
            end
            
            if drifting_precision == 0
                % Calculate the Doppler profile. We reuse "aoa_c" to save some
                % memory.
                aoa_c = reshape( cos(aoa_c+pi).*cos(eoa_c) ,1,n_paths );
            end
            
            % The main loop to calculate the channel coefficients
            for i_rx = 1 : n_rx                               % Rx elements
                for i_tx = 1 : n_tx                           % Transmit elements
                    ind = (i_tx-1)*n_rx + i_rx;               % Index of element in c
                    
                    % Get the antenna pattern response
                    if drifting_precision >= 2
                        PatTx = [ reshape( Vt(1,:,:,i_rx,i_tx) , 1,n_paths ) ;...
                            reshape( Ht(1,:,:,i_rx,i_tx) , 1,n_paths ) ];
                        PatRx = [ reshape( Vr(1,:,:,i_rx,i_tx) , 1,n_paths ) ;...
                            reshape( Hr(1,:,:,i_rx,i_tx) , 1,n_paths ) ];
                    else
                        PatTx = [ reshape( Vt(1,:,:,i_tx) , 1,n_paths ) ;...
                            reshape( Ht(1,:,:,i_tx) , 1,n_paths ) ];
                        PatRx = [ reshape( Vr(1,:,:,i_rx) , 1,n_paths ) ;...
                            reshape( Hr(1,:,:,i_rx) , 1,n_paths ) ];
                    end
                    
                    if use_polarization_rotation > 0
                        
                        % Get the LOS polarization rotation matrix
                        %   Rp = [ cos(deg(r,t))  ,  sin(deg(r,t)) ;...
                        %          -sin(deg(r,t)) ,  cos(deg(r,t)) ];
                        % This is represented as a vector with indices
                        % [1,3 ; 2,4] -> [1;2;3;4]   (result of reshaping )
                        
                        xprmat(1,:) = cos(degu(ind,:));
                        xprmat(2,:) = -sin(degu(ind,:));
                        xprmat(3,:) = -xprmat(2,:);
                        xprmat(4,:) = xprmat(1,:);
                        
                        % When we want elliptical polarization, we scale the
                        % H-componenet by an offset phase
                        if use_polarization_rotation == 2
                            pow_pat(ind,:,i_snapshot) = sum( [ sum( PatTx .* xprmat([1 3],:)) ;...
                                sum( PatTx .* xprmat([2 4],:))] .* PatRx );
                            
                            xprmat([3,4],:) = xprmat([3,4],:) .* kappa([1,1],:);
                        end
                    else
                        % The pattern without polarization (WINNER method)
                        pow_pat(ind,:,i_snapshot) = sum( PatTx .* PatRx );
                    end
                    
                    % Get the channel coefficients:
                    c(ind,:,i_snapshot) = sum( [ sum( PatTx .* xprmat([1 3],:)) ;...
                        sum( PatTx .* xprmat([2 4],:))] .* PatRx );
                    
                    % The phases
                    switch drifting_precision
                        case 0
                            % If drifting is disabled, the arriving angles do not change. We thus
                            % only need to update the rotating phasors.
                            cp(ind,:,i_snapshot) = exp( -1j*( pin +...
                                wave_no*( Pt(1,:,i_tx) + Prn(1,:,i_rx) ) -...
                                CPt(1,:,i_tx) - CPr(1,:,i_rx)  ));
                            
                            % Only one snapshot is calculated, the others are
                            % emultes by phase rotation.
                            c(ind,:,:) = c(ind,:,ones(1,n_snapshots) );
                            for ss = 2:n_snapshots
                                cp(ind,:,ss) = cp(ind,:,1) .* exp( -1j *...
                                    wave_no * aoa_c * dist(ss) );
                            end
                            
                        case 1
                            % In drifting mode, we have to update the coefficient
                            % matrix with the time-variant Doppler profile.
                            cp(ind,:,i_snapshot) = exp( -1j*( pin +...
                                wave_no*( Pt(1,:,i_tx) + Prn(1,:,i_rx) ) -...
                                CPt(1,:,i_tx) - CPr(1,:,i_rx) +...
                                phase( i_snapshot,: ) ));
                            
                        otherwise  % case 2 and 3
                            % In high precision model get their phases from the
                            % precalculated values.
                            cp(ind,:,i_snapshot) = exp( -1j*( pin - ...
                                CPt(1,:,i_rx,i_tx) - CPr(1,:,i_rx,i_tx) +...
                                phase( i_snapshot , : , i_rx , i_tx )));
                    end
                end
            end
        end
        
        % When summing up over the 20 subpaths, the
        % random initial phases will destroy the power
        % information. This needs to be fixed. Therefore, we
        % need the phase of each subpath when no antenna
        % pattern is applied. From these values, we can get a scaling
        % factor.
        
        if use_polarization_rotation == 1
            pow_pat = c;
        end
        
        % Combine pattern and phase coefficients
        c = c .* cp;
        
        % Sum up over all 20 subpaths
        cn = sum( reshape( c , n_links , n_clusters , n_subpaths , n_snapshots ) ,3);
        
        % Calculate phases using the normalized pattern response
        % If pow_pat is 0, we use the values of cp instead
        if drifting_precision == 0
            pow_pat = pow_pat(:,:,ones(1,n_snapshots));
        end
        ind = pow_pat == 0;
        tmp = cp(ind);
        cp = c ./ pow_pat;
        cp(ind) = tmp;
        
        % The path powers
        p_cl = h_channel_builder.pow(i_mobile*ones(1,n_links),:,1,ones(1,n_snapshots));
        
        % The influence of the phases
        p_ph = sum( reshape( cp , n_links , n_clusters , n_subpaths , n_snapshots ) ,3);
        p_ph = mean( abs(p_ph).^2 , 4 );
        p_ph = p_ph(:,:,:,ones(1,n_snapshots));
        
        % The scaling coefficient
        p_sc = ( p_cl./p_ph ).^0.5;
        
        % Combine path powers with channel coefficients
        if use_subpath_output
            c = p_sc(:,:,ones(1,n_subpaths),:) .* reshape( c , n_links , ...
                n_clusters , n_subpaths , n_snapshots );
        else
            c = p_sc .* cn;
        end
        
        
        % Now we apply the K-Factor and the shadowing profile
        if drifting_precision > 0 && map_valid == 1
            
            % Get shadowing profile along the track from the correlation
            % map. The first vector is the K-Factor and the second vector
            % is the SF. The initial K-Factor is already applied in the
            % path powers. We thus need to correct for that factor.
            [sf,kf] = h_channel_builder.par.get_sf_profile( ...
                h_channel_builder.track(i_mobile) , i_mobile );
            
            % Scaling factor for the KF
            kf  = kf./h_channel_builder.par.kf(i_mobile);
            
            p1  = h_channel_builder.pow(1);
            kf_power_scale = sqrt( 1+p1*(kf-1) );
            
            kf = sqrt(kf);
            
            % The path loss might be given manually together with the SF in
            % the  track object. In this case, we do not calculate it here
            if isempty( h_channel_builder.track(i_mobile).par ) || ...
                isempty( h_channel_builder.track(i_mobile).par.pg )
                
                % Get the path loss
                [ path_loss , scale_sf ] = h_channel_builder.par.get_pl( ...
                    h_channel_builder.track(i_mobile) , i_mobile );
            else
                % No path loss model is used when PL/SF are defined
                % manually.
                path_loss = 0;
                scale_sf = 1;
            end
            
            % We have the option to calculate the SF, PL and KF
            % antenna-dependent. This is activated when
            % simpar.drifting_precisioin is set to 3. However, SF
            % depends only on the rx position. We compy the values for
            % the tx-antennas here.
            if drifting_precision == 3
                rx_power = -path_loss + 10*log10( sf(:,:,o_tx) ) .* scale_sf;
            else
                rx_power = -path_loss + 10*log10( sf ) .* scale_sf;
            end
            rx_power = sqrt( 10.^( 0.1 * rx_power ) );
            
            if drifting_precision == 3
                kf = permute( reshape( kf(:,:,o_tx) , n_snapshots , [] ) , [2,3,4,1] );
                rx_power = rx_power ./ kf_power_scale(:,:,o_tx);
                rx_power = permute( reshape( rx_power , n_snapshots , [] ) , [2,3,4,1] );
            else
                o_tmp = ones(1,n_tx*n_rx);
                kf = permute( kf(:,o_tmp) , [2,3,4,1] );
                rx_power = rx_power ./ kf_power_scale;
                rx_power = permute( rx_power(:,o_tmp) , [2,3,4,1] );
            end
            
            % Apply the rx-power values.
            if use_subpath_output
                c(:,1,:,:) = c(:,1,:,:).*kf(:,:,o_sub,:);
                c = c.*rx_power(:,o_clusters,o_sub,:);
            else
                c(:,1,:,:) = c(:,1,:,:).*kf;
                c = c.*rx_power(:,o_clusters,1,:);
            end
            
        else
            % The initial KF is already applied in path powers. Here, we only
            % need to apply the SF and the path loss.
            c = c * rx_power(i_mobile);
        end
        
        % Apply antenna coupling
        
        Ct = h_channel_builder.tx_array.coupling;
        Cr = h_channel_builder.rx_array(i_mobile).coupling.';       % Conjugate ???
        
        if use_subpath_output
            
            % Reshape objects
            c = reshape( c , n_rx , n_tx , n_clusters , n_subpaths , n_snapshots );
            c = permute( c , [1,2,3,5,4] );
            
            % Apply the antenna coupling
            d = zeros( size(Cr,1) , size(Ct,2) , n_clusters , n_snapshots , n_subpaths  );
            for i_snapshot = 1:n_snapshots
                for i_cluster = 1:n_clusters
                    for ll = 1:n_subpaths
                        d(:,:,i_cluster,i_snapshot,ll) = Cr * c(:,:,i_cluster,i_snapshot,ll) * Ct;
                    end
                end
            end
            
        else
            % Reshape objects
            c = reshape( c , n_rx , n_tx , n_clusters , n_snapshots );
            
            % Apply the antenna coupling
            d = zeros( size(Cr,1) , size(Ct,2) , n_clusters , n_snapshots  );
            for i_snapshot = 1:n_snapshots
                for i_cluster = 1:n_clusters
                    d(:,:,i_cluster,i_snapshot) = Cr * c(:,:,i_cluster,i_snapshot) * Ct;
                end
            end
            
            if drifting_precision >= 2
                % When we use high precision, the delays on all elements are
                % different. However, antenna coupling will merge the
                % coefficients of different antennas. This needs to be
                % considered by the delays too.
                
                % The delays on different elements are weighted by the
                % powers in the coupling matrix.
                
                Cr_dl = zeros( size( Cr ));
                for i_rx = 1:size( Cr,1 )
                    tmp = abs( Cr( i_rx , : ) ).^2;
                    Cr_dl( i_rx , : ) = tmp./sum(tmp);
                end
                
                Ct_dl = zeros( size( Ct ));
                for i_tx = 1:size( Ct,2 )
                    tmp = abs( Ct( : , i_tx ) );
                    Ct_dl( : , i_tx ) = tmp./sum(tmp);
                end
                
                % Here, we scale the delays for each path by the
                % coupling.powers.
                
                delay = permute( delay, [3,4,2,1] );
                dl = zeros( size(Cr,1) , size(Ct,2) , n_clusters , n_snapshots  );
                for i_snapshot = 1:n_snapshots
                    for i_cluster = 1:n_clusters
                        dl(:,:,i_cluster,i_snapshot) = Cr_dl * delay(:,:,i_cluster,i_snapshot) * Ct_dl;
                    end
                end
            end
        end
        
        % Save everything to output data structure
        if drifting_precision >= 2
            h_channel(i_mobile) = channel( d , dl , initial_pos , false  );
        else
            h_channel(i_mobile) = channel( d , delay' , initial_pos , false  );
        end
        h_channel(i_mobile).name = [ h_channel_builder.name ,'_', h_channel_builder.track(i_mobile).name ];
        h_channel(i_mobile).rx_position = h_channel_builder.track(i_mobile).positions_abs;
        h_channel(i_mobile).tx_position = h_channel_builder.par.tx_position;
    end
    
end

if verbose && nargin == 1
    fprintf('] %5.0f seconds\n',round( etime(clock, tStart) ));
end

end

