classdef channel_builder < handle
%CHANNEL_BUILDER Class for generating the channel coefficients
%
% DESCRIPTION
% This class implements all functions that are needed to generate the
% channel coefficients. It thus implements the core components of the
% channel model. The class holds all the input variables as properties.
% It's main function 'get_channels' then generates the coefficients. The
% procedure is summarized as follows:
%
% The channel builder first generates a set of random clusters around each
% receiver. This is done by drawing random variables for the delay, the
% power and the departure and arrival angles for each cluster. Each cluster
% thus represents the origin of a reflected (and scattered) signal. The
% clusters are then represented as taps in the final CIR. The random
% variables fit the distributions and correlations defined by the
% parameter_set object.
%
% Next, antenna dependent parameters are extracted for each user. Those
% depend on the position of the terminal, it's orientation and the equipped
% antennas. The polarization rotation of the NLOS taps is modeled by a
% random variable which fits to the  distribution defined by the
% parameter_set. The LOS polarization is calculated from the geometric
% orientation of the antennas. A core function here is the interpolation of
% the antenna patterns which results in a specific H and V value for each
% subpath.
%
% The core function then generated the coefficients themselves. This is
% done for each antenna element and for each snapshot separately and
% also includes the Doppler shift of each subpath. Finally, the k-factor
% and the shadow fading are applied and a all the data is returned as an
% object of class channel.
%
% REFERENCE
% Some functions were taken from the Winner channel model. "Kyösti, P.;
% Meinilä, J.; Hentilä, L. & others; {IST-4-027756 WINNER II D1.1.2 v.1.1}:
% WINNER II Channel Models; 2007". New functions support the geometric
% generation of polarized channels and non-linear tracks.
%
%
% channel_builder Properties:
%    name - Name of the channel builder
%    taus - Initial delays
%    pow - Initial powers
%    AoD - Initial azimuth of departure angles
%    AoA - Initial azimuth of arrival angles
%    EoD - Initial elevation of departure angles
%    EoA - Initial elevation of departure angles
%    xpr - Initial cross polarization power ratio for the NLOS components
%    pin - Initial phases (for each path and subpath)
%    kappa - Polarization phases
%    subpath_coupling - Random coupling of subpaths
%    simpar - Object of class simulation_parameters
%    par - Object of class parameter_set
%    track - Track for each Rx
%    rx_array -  Antenna array of each Rx
%    tx_array - Antenna array of each Tx
%
% channel_builder Methods:
%    get_channels - Generates channel coefficients
%    winner_compatible_bulkpar - Create Winner-compatible output data structures
%    correction_function - Corrects the initial RMS Angular spread
%    generate_initial_angles - Generate angular parameters (private)
%    generate_initial_paths - Generate delays and powers (private)
%    generate_xpr - Generate XPR (private)
%    get_drifting - Generate drifting angles and delays (private)
%    get_subpath_angles - Generate subpaths and perform random coupling (private)
%
%
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% Fraunhofer Heinrich Hertz Institute
% Wireless Communication and Networks
% Einsteinufer 37, 10587 Berlin, Germany
%  
% This file is part of QuaDRiGa.
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% QuaDRiGa is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%     
% You should have received a copy of the GNU Lesser General Public License
% along with QuaDRiGa. If not, see <http://www.gnu.org/licenses/>.

    properties
        name = 'channel';                   % Name of the channel builder
        taus                                % Initial delays
        pow                                 % Initial powers
        AoD                                 % Initial azimuth of departure angles
        AoA                                 % Initial azimuth of arrival angles
        EoD                                 % Initial elevation of departure angles
        EoA                                 % Initial elevation of departure angles
        xpr                                 % Initial cross polarization power ratio for the NLOS components
        pin                                 % Initial phases
        kappa                               % Polarization phases
        subpath_coupling                    % Random coupling of subpaths
    end
    
    properties(Dependent)
        simpar                              % Object of class simulation_parameters
        par                                 % Object of class parameter_set
        track                               % Track for each Rx
        rx_array                            % Antenna array of each Rx
        tx_array                            % Antenna array of each Tx
    end
    
    properties(Access = private)
        Psimpar = simulation_parameters;
        Ppar = parameter_set;
        Ptrack = track;
        Prx_array = array;
        Ptx_array = array;
        pow_wo_kf
    end
    
    methods
        % Constructor
        function obj = channel_builder( par )
            if numel( par ) > 1
                error('"par" must be scalar')
            end
                        
            obj.par         = par;
            obj.simpar      = par.simpar;
            obj.tx_array    = par.tx_array;
            obj.name        = par.name;
            
            if par.no_positions > 1 && numel( par.track ) == 1
                obj.track   = par.track(ones(1,par.no_positions));
            else 
                obj.track   = par.track;
            end
            
            if par.no_positions > 1 && numel( par.rx_array ) == 1
                obj.rx_array = par.rx_array(ones(1,par.no_positions));
            else 
                obj.rx_array = par.rx_array;
            end
        end
        
        
        % Get-Functions
        function out = get.simpar(obj)
            out = obj.Psimpar;
        end
        function out = get.par(obj)
            out = obj.Ppar;
        end
        function out = get.track(obj)
            out = obj.Ptrack;
        end
        function out = get.rx_array(obj)
            out = obj.Prx_array;
        end
        function out = get.tx_array(obj)
            out = obj.Ptx_array;
        end
        
        
        % Set-Functions
        function set.name(obj, value)
            if ~(ischar(value))
                error('??? "name" must be a string.')
            end
            obj.name = value;
        end
        
        function set.simpar(obj, value)
            if ~(isa(value, 'simulation_parameters'))
                error('??? "simpar" must be of class "simulation_parameters".')
            elseif ~all(size(value) == [1, 1])
                error('??? "simpar" must be scalar.')
            end
            obj.Psimpar = value;
        end
        
        function set.par(obj, value)
            if ~(isa(value, 'parameter_set'))
                error('??? "par" must be of class "parameter_set".')
            elseif ~all(size(value) == [1, 1])
                error('??? "par" must be scalar.')
            end
            obj.Ppar = value;
        end
        
        function set.track(obj, value)
            if ~(isa(value, 'track'))
                error('??? "track" must be of class "track".')
            elseif ~any(size(value) == 1)
                error('??? "track" must be vector.')
            elseif numel(value) ~= obj.Ppar.no_positions
                error('??? "track" must have as much elements as there are positions in "par".')
            end
            
            if size(value, 1) ~= 1
                value = value';
            end
            
            for n = 1:obj.Ppar.no_positions
                if ~(value(n).no_segments <= 2)
                    error('??? Each "track" must have at most 2 segments. Use "track.get_subtrack" to get subtracks.')
                end
            end
            obj.Ptrack = value;
        end
        
        function set.rx_array(obj, value)
            if ~(isa(value, 'array'))
                error('??? "rx_array" must be of class "array".')
            elseif ~any(size(value) == 1)
                error('??? "rx_array" must be vector.')
            elseif numel(value) ~= obj.Ppar.no_positions && numel(value) ~= 1
                error(['??? "rx_array" must either have as much elements as there', ...
                    ' are positions in "par" or only one entry.'])
            end
            
            if size(value, 1) ~= 1
                value = value';
            end
            
            if numel(value) == 1 && obj.Ppar.no_positions > 1
                for n = 2:obj.Ppar.no_positions
                    value(n) = value(1);                % We use handle classes --> no copying
                end
            end
            
            obj.Prx_array = value;
        end
        
        function set.tx_array(obj, value)
            if ~(isa(value, 'array'))
                error('??? "tx_array" must be of class "array".')
            elseif ~all(size(value) == [1, 1])
                error('??? "tx_array" must be scalar.')
            end
            obj.Ptx_array = value;
        end
    end
    
    methods(Static)
        h_channel = get_los_channels( h_parset )
    end
end

