function c = interpolate( obj , dist , track_length , method )
%INTERPOLATE Interpolates the channel
%
% The channel builder creates one snapshot for each position that is listed
% in the track object. When the channel sampling theorem is not violated
% (i.e. the sample density is >= 2), then the channel can be interpolated
% to any other position on the track. This can be used e.g. to emulate
% arbitrary movements along the track.
%
%    c = INTERPOLATE( dist , track_length ) interpolates the channel to the
%    positions given in dist. 'dist' must be a vector containing distance
%    values on the track. The distance is measured in m from the beginning
%    of the track. 'track_length' is the total length of the track. 'c' is
%    new object of class channel.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

if nargin < 3
    error('??? wrong number of input arguments');
end

if nargin == 3
    method = 'linear';
else
    supported_methods = {'linear','spline'};
    if ~( ischar(method) && any( strcmp(method,supported_methods)) )
        str = 'Interpolation method type not found; supported are: ';
        no = numel(supported_methods);
        for n = 1:no
            str = [str,supported_methods{n}];
            if n<no
                str = [str,', '];
            end
        end
        error(str);
    end
end

if ~( isnumeric(track_length) &&...
        isreal(track_length) &&...
        all(size(track_length) == 1) &&...
        track_length>0  )
    error('??? "track_length" must me numeric, scalar and >0');
end

if ~( isnumeric(dist) &&...
        isreal(dist) &&...
        any(size(dist) == 1) &&...
        min(dist)>=0 && max(dist) <= track_length )
    error('??? "dist" must me numeric and can not exceed track length');
end

% Get the dimension of the channel tensor
nrx = obj.no_rx;
ntx = obj.no_tx;
nsnap = obj.no_snap;
L = obj.no_path;
ndist = numel( dist );
individual_delays = obj.individual_delays;

% Get the positions of the snapshots in the original channel
pos_snap = (0:nsnap-1) * track_length/(nsnap-1);
pos_snap(end) = track_length;

switch method
    
    case 'linear'
        
        % Get the variables for the linear interpolation
        [tmp,b] = sort( dist );
        [~,a]   = sort( [pos_snap,tmp] );
        ui      = 1:(nsnap + ndist);
        ui(a)   = ui;
        ui      = ui(nsnap+1:end) - (1:ndist);
        ui(b)   = ui;
        ui( ui==nsnap ) = nsnap-1;
        uin     = ui+1;
        u       = (dist-pos_snap(ui))./( pos_snap(uin)-pos_snap(ui) );
        
        % Interpolate the channel and write data to a new channel object
        c = channel;
        c.name = obj.name;
        c.version = obj.version;
        c.individual_delays = obj.individual_delays;
        
        % Expand the array size
        v = u( ones( 1, nrx*ntx*L ) ,:);
        v = reshape( v , nrx,ntx,L,ndist );
        
        % Interpolate amplitudes and phases separately
        c1 = (1-v).*abs(obj.coeff(:,:,:,ui)) + v.*abs(obj.coeff(:,:,:,uin));
        c2 = angle( (1-v).*obj.coeff(:,:,:,ui) + v.*obj.coeff(:,:,:,uin) );
        c.coeff = c1.*exp(1j*c2);
        
        if individual_delays
            c.delay = (1-v).*obj.delay(:,:,:,ui) + v.*obj.delay(:,:,:,uin);
        else
            v = u( ones( 1, L ) ,:);
            c.delay = (1-v).*obj.delay(:,ui) + v.*obj.delay(:,uin);
        end
        
        % Interpolate rx positions (if provided)
        if ~isempty(obj.rx_position)
            c.rx_position = zeros( 3,ndist );
            for n=1:3
                c.rx_position(n,:) = (1-u).*obj.rx_position(n,ui) + u.*abs(obj.rx_position(n,uin));
            end
        end
        
        
    case 'spline'

        c = channel;
        c.name = obj.name;
        c.version = obj.version;
        c.individual_delays = individual_delays;
        c.coeff = zeros( nrx,ntx,L,ndist );
        
        % Here, we use the MATLAB internal spline interpolation
        for r = 1:nrx
            for t = 1:ntx
                for l = 1:L
                    c0 =  reshape( obj.coeff(r,t,l,:) , 1, nsnap ) ;
                    c1 = spline( pos_snap , abs(c0) , dist );
                    c2 = angle( spline( pos_snap , c0 , dist ) );
                    c.coeff(r,t,l,:) = c1.*exp(1j*c2);
                    
                    if individual_delays
                        c.delay(r,t,l,:) = ...
                            pchip( pos_snap , reshape( obj.delay(r,t,l,:) ,1,nsnap ) , dist );
                    end
                end
            end
        end
        
        if ~individual_delays
            for l = 1:L
                c.delay(l,:) = pchip( pos_snap , obj.delay(l,:) , dist );
            end
        end
        
        % Interpolate rx positions (if provided)
        if ~isempty(obj.rx_position)
            c.rx_position = zeros( 3,ndist );
            for n=1:3
                c.rx_position(n,:) = pchip( pos_snap , obj.rx_position(n,:) , dist );
            end
        end
        
    otherwise
        return
end

% Copy tx position (if provided)
if ~isempty(obj.tx_position)
   c.tx_position = obj.tx_position;
end

end

