function frequency_response = fr(h_channel, B, n_carriers, i_snapshot)
%FR Calculates the frequency response of the channel
%
%   frequency_response = h_channel.fr(B, n_carriers) calculates the frequency response of the channel "h_channel" for the
%   given bandwidth "B" at "n_carriers" equally spaced carriers. The size of the frequency_response is given by
%   [n_rx, n_tx, n_carriers, n_i_snapshots].
%
%   frequency_response = h_channel.fr(B, pilot_grid) calculates the frequency response of the channel "h_channel" for the
%   given bandwidth "B" at the pilot grid specified by "pilot_grid". "pilot_grid" must be a
%   vector containing values between 0 and 1.
%
%   frequency_response = h_channel.fr(B, n_carriers, i_snapshot) calculates the frequency response at the snapshots given by "i_snapshot".
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input arguments

if numel(h_channel) == 1
    check = true;
    if nargin < 3
        error('??? You need to specify the bandwidth ''B'' and the number of carriers ''n_carriers''.')
    elseif nargin < 4
        i_snapshot = 1:h_channel.no_snap;
        check = false;
    end
    
    if ~( size(B,1) == 1 && isnumeric(B) && all(size(B) == [1 1]) && min(B) > 0 )
        error('??? The bandwidth "B" must be scalar and > 0')
    end
    
    if isnumeric(n_carriers) && isreal(n_carriers)
        if all(size(n_carriers) == [1 1]) && mod(n_carriers,1)==0 && n_carriers>0
            pilot_grid = ( 0:n_carriers-1 )/n_carriers;
        elseif numel( size(n_carriers) ) == 2 && any(size(n_carriers)==1)
            if size(n_carriers,2) == 1
                pilot_grid = n_carriers.';
            else
                pilot_grid = n_carriers;
            end
            n_carriers = numel(pilot_grid);
        else
            error('??? Invalid input for "n_carriers" or "pilot_grid"')
        end
    else
        error('??? The no. of carriers "n_carriers" must be numeric')
    end
    
    if check
        if ~( any( size(i_snapshot)==1 ) && isnumeric(i_snapshot) && all( mod(i_snapshot,1)==0 ) ...
                && min(i_snapshot) > 0 && max(i_snapshot)<=h_channel.no_snap )
            error('??? The snapshot range must be numeric, integer and can not exceed the numbers of snapshots');
        end
    end
    
    % Get the dimension of the channel tensor
    n_rx = h_channel.no_rx;
    n_tx = h_channel.no_tx;
    n_i_snapshots = numel(i_snapshot);
    n_taps = h_channel.no_path;
    
    % Preallocate some memory and rearrange coefficients
    frequency_response = zeros(n_i_snapshots * n_rx * n_tx, n_carriers);
    if h_channel.individual_delays
        m = reshape(permute(h_channel.delay(:, :, :, i_snapshot)*B, [ 4 1 2 3 ]), n_i_snapshots*n_rx*n_tx, n_taps);
    else
        m = repmat(h_channel.delay(:, i_snapshot)'*B, n_rx*n_tx, 1);
    end
    c = reshape(permute(h_channel.coeff(:, :, :, i_snapshot), [ 4 1 2 3 ]), n_i_snapshots*n_rx*n_tx, n_taps);
    
    % The arguments of the exponential function
    v = -2 * pi * 1j * pilot_grid;
    
    % The main calculation
    o_carriers = ones(1, n_carriers);
    for i_tap = 1:n_taps
        frequency_response = frequency_response + c(:, o_carriers*i_tap) .* exp(m(:, i_tap) * v);
    end
    
    % Reorder the output dimensions
    frequency_response = reshape(frequency_response, n_i_snapshots, n_rx, n_tx, n_carriers);
    frequency_response = permute(frequency_response, [ 2 3 4 1 ]);
    
else
    frequency_response = cell(size(h_channel));
    if nargin == 3
        for i_channel = 1:numel(h_channel)
            frequency_response{i_channel} = h_channel(i_channel).fr(B, n_carriers);
        end
    else
        for i_channel = 1:numel(h_channel)
            frequency_response{i_channel} = h_channel(i_channel).fr(B, n_carriers, i_snapshot);
        end
    end
end

end

