classdef channel < handle & matlab.mixin.Copyable
%CHANNEL Class for handling channel coefficients
%
% DESCRIPTION
% This class combines all functions for storing and handling channel
% coefficients. Channel objects are created by the channel_builder.
%
%
% channel Properties:
%    name - Identifyer of the channel object
%    version - Version number
%    coeff - Coefficient tensor
%    delay - Delay matrix
%    initial_position - Index of the initial position on the track
%    tx_position - Position of the transmitter
%    rx_position - Position of the receiver
%    no_rx - Number of receive elements (read only)
%    no_tx - Number of transmit elements (read only)
%    no_path - Number of taps (read only)
%    no_snap - Number of snapshots (read only)
%
% channel Methods:
%    copy_objects - Creates a copy of the current channel object
%    fr - Calculates the frequency response of the channel
%    interpolate - Interpolates the channel
%    merge - Combine channel segments into a continuous time evolution channel
%
%
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% Fraunhofer Heinrich Hertz Institute
% Wireless Communication and Networks
% Einsteinufer 37, 10587 Berlin, Germany
%  
% This file is part of QuaDRiGa.
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% QuaDRiGa is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%     
% You should have received a copy of the GNU Lesser General Public License
% along with QuaDRiGa. If not, see <http://www.gnu.org/licenses/>.
    
    properties
        name = 'New channel';       % Identifier of the channel object
        version = simulation_parameters.version;        % Version number
    end
    
    properties(Dependent)
        individual_delays           % Different delays for each Rx/Tx antenna
        coeff                       % Coefficient tensor
        delay                       % Delay matrix
        initial_position            % Index of the initial position on the track
        tx_position                 % Position of the transmitter
        rx_position                 % Position of the receiver
    end
    
    % Informative parameters
    properties(Dependent,SetAccess=protected)
        no_rx                       % Number of receive elements (read only)
        no_tx                       % Number of transmit elements (read only)
        no_path                     % Number of taps (read only)
        no_snap                     % Number of snapshots (read only)
    end
    
    % Data storage
    properties(Access=private)
        Pindividual_delays  = false;
        Pcoeff              = [];
        Pdelay              = [];
        Pinitial_position   = 0;
        Ptx_position        = [];
        Prx_position        = [];
    end
    
    methods
        % Constructor
        function obj = channel( Ccoeff , Cdelay , Cinitial_position , ~ )
            check = true;
            if nargin == 4
                check = false;
            end
            if nargin > 0
                if check
                    obj.coeff = Ccoeff;
                else
                    obj.Pcoeff = Ccoeff;
                end
            end
            if nargin > 1
                if  numel( size( Cdelay ) ) == numel( size( Ccoeff ) ) && ...
                        all( size( Cdelay ) == size( Ccoeff ) ) && ...
                        numel( size( Cdelay ) ) > 2
                    obj.individual_delays = 1;
                end
                if check
                    obj.delay = Cdelay;
                else
                    obj.Pdelay = Cdelay;
                end
            end
            if nargin > 2
                if check
                    obj.initial_position = Cinitial_position;
                else
                    obj.Pinitial_position = Cinitial_position;
                end
            end
        end
        
        % Destructor
        function delete(obj)
            obj.Pindividual_delays  = false;
            obj.Pcoeff              = [];
            obj.Pdelay              = [];
            obj.Pinitial_position   = 0;
            obj.Ptx_position        = [];
            obj.Prx_position        = [];
        end
        
        % Get functions
        function out = get.individual_delays(obj)
            out = obj.Pindividual_delays;
        end
        function out = get.coeff(obj)
            out = obj.Pcoeff;
        end
        function out = get.delay(obj)
            out = obj.Pdelay;
        end
        function out = get.initial_position(obj)
            out = obj.Pinitial_position;
        end
        function out = get.rx_position(obj)
            out = obj.Prx_position;
        end
        function out = get.tx_position(obj)
            out = obj.Ptx_position;
        end
        function out = get.no_rx(obj)
            out = size( obj.coeff,1);
        end
        function out = get.no_tx(obj)
            out = size( obj.coeff,2);
        end
        function out = get.no_path(obj)
            s = size(obj.coeff);
            if numel(s) < 3
                if any(s)==0
                    out = 0;
                else
                    out=1;
                end
            else
                out = s(3);
            end
        end
        function out = get.no_snap(obj)
            s = size(obj.coeff);
            if numel(s) < 4
                if any(s)==0
                    out = 0;
                else
                    out=1;
                end
            else
                out = s(4);
            end
        end
        function v = get_version(obj)
            v = regexp( obj.version,'(?<a>[0-9]+).(?<b>[0-9]+).(?<c>[0-9]+)-(?<d>[0-9]+)' ,'tokens');
            v = str2double(v{1});
        end
        
        
        % Set functions
        function set.name(obj,value)
            if ~( ischar(value) )
                error('??? "name" must be a string.')
            end
            obj.name = value;
        end
        
        function set.version(obj,value)
            if ~( ischar(value) )
                error('??? "version" must be a string.')
            elseif isempty( regexp(value,'[0-9]+.[0-9]+.[0-9]+-[0-9]+', 'once') )
                error('??? "version" must be a version-string.')
            end
            obj.version = value;
        end
        
        function set.individual_delays(obj,value)
            if ~( all(size(value) == [1 1]) && isreal(value) )
                error('??? "individual_delays" must be numeric and scalar')
            end
            value = logical( value );
            if obj.Pindividual_delays && ~value && ~isempty(obj.Pdelay)
                obj.Pdelay = reshape( obj.Pdelay(1,1,:,:) , obj.no_path , obj.no_snap );
            elseif ~obj.Pindividual_delays && value && ~isempty(obj.Pdelay)
                obj.Pdelay = reshape( obj.Pdelay,1,1,obj.no_path,obj.no_snap );
                obj.Pdelay = obj.Pdelay( ones(1,obj.no_rx) , ones(1,obj.no_tx) , : , :  );
            end
            obj.Pindividual_delays = value;
        end
        
        function set.coeff(obj,value)
            if ~isnumeric(value)
                error('??? "coeff" must be numeric')
            end
            
            so = size(obj.Pcoeff);
            sn = size(value);
            if numel(so) == numel(sn) && all( so == sn )
                obj.Pcoeff = value;
            else
                obj.Pcoeff = value;
                if obj.Pindividual_delays
                    obj.Pdelay = zeros( obj.no_rx , obj.no_tx , obj.no_path, obj.no_snap );
                else
                    obj.Pdelay = zeros( obj.no_path , obj.no_snap );
                end
                obj.Pinitial_position = 1;
            end
        end
        
        function set.delay(obj,value)
            a = size(value,1);
            b = size(value,2);
            c = size(value,3);
            d = size(value,4);
            if ~isnumeric(value)
                error('??? "delay" must be numeric')
            elseif obj.individual_delays && ...
                    ~( all( [a,b,c,d] == [ obj.no_rx , obj.no_tx , obj.no_path , obj.no_snap ] ) || ...
                    any( [a,b,c,d]==0 ) )
                error('??? "delay" must match the number of antennas, paths and snapshots')
            elseif ~obj.individual_delays && ...
                    ~all( [a,b] == [ obj.no_path , obj.no_snap ] )
                error('??? "delay" must match the number of paths and snapshots')
            end
            obj.Pdelay = value;
        end
        
        function set.initial_position(obj,value)
            if ~( all(size(value) == [1 1]) && isnumeric(value) ...
                    && isreal(value) && mod(value,1)==0 && value >= 0 )
                error('??? "initial_position" must be integer and >= 0')
            elseif value > obj.no_snap
                error('??? "initial_position" must not exceed the number of snapshots')
            end
            obj.Pinitial_position = value;
        end
        
        function set.rx_position(obj,value)
            if isempty( value)
                obj.Prx_position = [];
            else
                if ~( isnumeric(value) && isreal(value) && size(value,2) > 0 )
                    error('QuaDRiGa:Channel:wrongInputValue','??? "rx_position" must consist of real numbers')
                elseif ~all( size(value,1) == 3 )
                    error('QuaDRiGa:Channel:wrongInputValue','??? "rx_position" must have three rows.')
                elseif size( value,2 ) ~= obj.no_snap
                    error('QuaDRiGa:Channel:wrongInputValue','??? "rx_position" must match the number of snapshots.')
                end
                obj.Prx_position = value;
            end
        end
        
        function set.tx_position(obj,value)
            if isempty( value)
                obj.Ptx_position = [];
            else
                if ~( isnumeric(value) && isreal(value) )
                    error('QuaDRiGa:Channel:wrongInputValue','??? "tx_position" must consist of real numbers')
                elseif ~all( size(value,1) == 3 && size(value,2) == 1 )
                    error('QuaDRiGa:Channel:wrongInputValue','??? "tx_position" must have 3 rows')
                end
                obj.Ptx_position = value;
            end
        end
    end
   
end

