function han = visualize(obj, element_indices, han)
%VISUALIZE Create a plot showing the element configurations
%
%   VISUALIZE plots the 3D beam patterns of all antenna elements within the
%             antenna array.
%   VISUALIZE(element) only plots the beam patterns of the antenna elements 
%             specified by the given element indices contained in element_indices.
%
% You can change the the titles by manipulating the output handles.
%    set(han(1,10),'String','Element New')
%    set(han(1,11),'String','Vertical-New')
%    set(han(1,12),'String','Horizontal-New')
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

if nargin == 3
    update_plot = true;
else
    update_plot = false;
end

if nargin == 1
    element_indices = 1:obj.no_elements;
end

if ~( any(size(element_indices) == 1) && isnumeric(element_indices) ...
        && isreal(element_indices) && all(mod(element_indices,1)==0) && all(element_indices > 0) )
    error('??? "element_indices" must be integer and > 0')
elseif any( element_indices > obj.no_elements )
    error('??? "element_indices" exceed "no_elements"')
end



theta = obj.azimuth_grid;
if numel( theta ) == 361
    indt = 1:5:361;
else
    indt = 1:numel( theta );
end
theta = theta( indt );

phi = obj.elevation_grid';
if numel( phi ) == 181
    indp = 1:5:181;
else
    indp = 1:numel( phi );
end
phi = phi( indp );




for n = 1:numel(element_indices);
    
    patv =  obj.field_pattern_vertical( indp,indt,element_indices(n) ) ;
    path =  obj.field_pattern_horizontal( indp,indt,element_indices(n) ) ;
    
    if update_plot
        gain = han(n,9);
    else
        gain = max(abs([ reshape( patv,[],1 ) ; reshape( path,[],1 ) ]));
    end
    
    patv = patv/gain;
    path = path/gain;
    
    [theta1, phi1] = meshgrid(theta, phi);
    [Xi, Yi, Zi] = sph2cart(theta1, phi1, 1);
    clear phi1 theta1;
    % Xi = cos(phi)*cos(theta);
    % Yi = cos(phi)*sin(theta);
    % Zi = sin(phi)*ones(size(theta));
    
    B = obj.pol_vector(:,element_indices(n));
    B = B*24;
    
    if ~update_plot
        
        fs = 16;
        han(n,1) = figure('Position',[ 100 , 100 , 400 , 850]); hold on; axis off;
        han(n,10) = title(['Element ',num2str(element_indices(n))],'FontName','Times','FontSize',fs+4);
        
        han(n,2)=axes('position',[0.1 0.55 0.85 0.32]);
        C = 10*log10(abs(  patv  ).^2)+20; C(C<0)=0;
        X = C.*Xi; Y = C.*Yi; Z = C.*Zi;
        
        han(n,3) = surf(X,Y,Z,C);
        hold on
        han(n,4) = plot3( [0,B(1)],[0,B(2)],[0,B(3)],'k','Linewidth',3 );
        hold off
        
        axis([-20 20 -20 20 -20 20])
        caxis([0 20])
        set(gca,'xtick',-10:10:10);
        set(gca,'ytick',-10:10:10);
        set(gca,'ztick',-10:10:10);
        xlabel('x')
        ylabel('y')
        zlabel('z')
        han(n,11) = title('Vertical','FontName','Times','FontSize',fs);
        view(45, 33);
        set(gca,'FontName','Times','FontSize',fs)
        
        han(n,5)=axes('position',[0.1 0.14 0.85 0.32]);
        C = 10*log10(abs( path ).^2)+20; C(C<0)=0;
        X = C.*Xi; Y = C.*Yi; Z = C.*Zi;
        
        han(n,6) = surf(X,Y,Z,C);
        hold on
        han(n,7) = plot3( [0,B(1)],[0,B(2)],[0,B(3)],'k','Linewidth',3 );
        hold off
        
        axis([-20 20 -20 20 -20 20])
        caxis([0 20])
        set(gca,'xtick',-10:10:10);
        set(gca,'ytick',-10:10:10);
        set(gca,'ztick',-10:10:10);
        xlabel('x')
        ylabel('y')
        zlabel('z')
        han(n,12) = title('Horizontal','FontName','Times','FontSize',fs);
        view(45, 33);
        set(gca,'FontName','Times','FontSize',fs)
        
        han(n,8)=axes('position',[0.08 0.08 0.08 0.08]);
        set(han(n,8), 'Visible', 'Off');
        
        g = 10*log10(gain^2);
        caxis([g-20,g]);
        h=colorbar('SouthOutside','XTick',floor(g)-18:3:floor(g));
        set(h,'position',[0.08 0.07 0.80 0.03])
        set(gca,'FontName','Times','FontSize',fs)
        zlab = get(h,'xlabel');
        set(zlab,'String','Attenuation (dB)','FontName','Times','FontSize',fs);
        
        han(n,9) = gain;
        
    else
        
        C = 10*log10(abs(  patv  ).^2)+20; C(C<0)=0;
        X = C.*Xi; Y = C.*Yi; Z = C.*Zi;
        set( han(n,3) , 'Xdata', X ,  'Ydata', Y , 'Zdata', Z , 'Cdata',C);
        
        C = 10*log10(abs( path ).^2)+20; C(C<0)=0;
        X = C.*Xi; Y = C.*Yi; Z = C.*Zi;
        set( han(n,6) , 'Xdata', X ,  'Ydata', Y , 'Zdata', Z , 'Cdata',C);
        
        set( han(n,4) , 'Xdata', [0,B(1)] ,  'Ydata', [0,B(2)] , 'Zdata', [0,B(3)] );
        set( han(n,7) , 'Xdata', [0,B(1)] ,  'Ydata', [0,B(2)] , 'Zdata', [0,B(3)] );
        
        drawnow
    end
    
    
end




end

