function [V,H,CP,dist] = interpolate( h_array , azimuth , elevation , element ,...
    azimuth_grid , elevation_grid ,...
    field_pattern_vertical , field_pattern_horizontal , common_phase , element_position )
%INTERPOLATE Interpolates the field pattern
%
%   [V,H] = INTERPOLATE( azimuth,elevation ) Interpolates the field pattern to the
%   given azimuth and elevation values. Azimuth and elevation are angles in the
%   spheric coordinate system. Azimuth can range from -Pi to Pi and elevation can
%   have values in between -Pi/2 and Pi/2. Values outside this range will be
%   converted accordingly. Important: All angles must be given in radians!
%
%   [V,H] = INTERPOLATE( azimuth,elevation,element ) also specifies the element
%   range for which the pattern will be interpolated.
%
%   [V,H,dist] = INTERPOLATE( azimuth,elevation,element ) also returns the
%   effective distances between the antenna elements when seen from the
%   direction of the incident path. The distance is calculated by an
%   projection of the array positions on the normale plane of the incident
%   path.
%
%   Note: Interpolation of the beam patterns is very (!!!) computing intensive. It
%   must be performed several thousands of times during a simulation run. It is
%   highly recommended to use nearest-neighbor interpolation for this task since
%   this method is optimized for speed. Linear and spline interpolation call the
%   MATLAB internal interpolation function which is more than 10 times slower.
%   To enable nearest-neighbor interpolation, set the 'interpolation_method'
%   property of the array object to 'nearest'.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input arguments

% Determine, if the common phase needs to be evaluated
calc_cp = true;
if nargout < 3
    calc_cp = false;
end

if nargin < 5
    if nargin < 3
        error('??? You need to specify azimuth and elevation for the pattern interpolation.');
    elseif nargin < 4
        element = 1:h_array.no_elements;
    end
    
    % Check element input
    if ~( size(element,1) == 1 && isnumeric(element) ...
            &&  all( mod(element,1)==0 ) && min(element) > 0 && max(element)<=h_array.no_elements )
        error('??? "element" must be integer > 0 and can not exceed array size')
    end
    
    % Check azimuth input
    if ~( isnumeric(azimuth) && isreal(azimuth) )
        error('??? "azimuth" must be numeric and real valued.')
    end
    dims = size( azimuth );
    
    % Check elevation input
    if ~( isnumeric(elevation) && isreal(elevation) && all( size(elevation) == dims ) )
        error('??? "elevation" must be numeric and real valued and it must match the size of "azimuth"')
    end
    
    azimuth_grid  = h_array.azimuth_grid;
    elevation_grid = h_array.elevation_grid;
    field_pattern_vertical = h_array.field_pattern_vertical;
    field_pattern_horizontal = h_array.field_pattern_horizontal;
    element_position = h_array.element_position(:,element);    
    
    if calc_cp
        common_phase = exp( 1j*h_array.common_phase );      % Input
    end
end

% Get initial values
dims = size( azimuth );
no_values = numel( azimuth );
no_element = numel( element );
no_az = h_array.no_az;
no_el = h_array.no_el;

% Reduce dimensions of arrays
azimuth = reshape( azimuth,1,no_values );
elevation = reshape( elevation,1,no_values );

% elevation may be outside of (-pi/2,pi/2).
% Mapping of angles to (-pi/2,pi/2) and changing the azimuthal orientation of rays
% with original elevation angles within +-(pi/2,3pi/2)+n*2*pi. 29.9.2008 EsaK

ind = floor( elevation/pi + 0.5 );
elevation = (-1).^(ind + 1).*(pi*ind - elevation);

ind = logical( mod(ind,2) );
azimuth(ind) = azimuth(ind) + pi;

azimuth = mod( azimuth+pi , 2*pi )-pi;

V  = zeros( no_values ,no_element );         % Initialize variables
H  = zeros( no_values ,no_element );

if calc_cp
    CP = zeros( no_values ,no_element );
end

% Interppolate field patterns
switch h_array.interpolation_method
    case 'nearest'
        
        % Determine the nearest location of xi in x
        [tmp,b] = sort( azimuth );
        [~,a]   = sort( [azimuth_grid,tmp] );
        ui      = 1:(no_az + no_values);
        ui(a)   = ui;
        ui      = ui(no_az+1:end) - (1:no_values);
        ui(b)   = ui;
        
        % Determine the nearest location of yi in y
        [tmp,b] = sort( elevation );
        [~,a]   = sort( [elevation_grid,tmp] );
        vi      = 1:(no_el + no_values);
        vi(a)   = vi;
        vi      = vi(no_el+1:end) - (1:no_values);
        vi(b)   = vi;
        
        % Determine the index of the element and get the corresponding pattern entry
        vi = vi+(ui-1)*no_el;
        for n=1:no_element
            ndx = (element(n)-1)*no_az*no_el + vi;
            V(:,n) = field_pattern_vertical(ndx);
            H(:,n) = field_pattern_horizontal(ndx);
            
            if calc_cp
                CP(:,n) = common_phase(ndx);
            end
        end
        
    case 'linear'
        
        % Determine the nearest location of xi in x and the difference to
        % the next point
        [tmp,b] = sort( azimuth );
        [~,a]   = sort( [azimuth_grid,tmp] );
        ui      = 1:(no_az + no_values);
        ui(a)   = ui;
        ui      = ui(no_az+1:end) - (1:no_values);
        ui(b)   = ui;
        ui( ui==no_az ) = no_az-1;
        uin     = ui+1;
        u       = (azimuth-azimuth_grid(ui))./( azimuth_grid(uin)-azimuth_grid(ui) );
        u       = u';
        
        % Determine the nearest location of yi in y and the difference to
        % the next point
        [tmp,b] = sort( elevation );
        [~,a]   = sort( [elevation_grid,tmp] );
        vi      = 1:(no_el + no_values);
        vi(a)   = vi;
        vi      = vi(no_el+1:end) - (1:no_values);
        vi(b)   = vi;
        vi( vi==no_el ) = no_el-1;
        vin     = vi+1;
        v       = (elevation-elevation_grid(vi))./( elevation_grid(vin)-elevation_grid(vi) );
        v       = v';
        
        % Calculate the scaling coefficients
        c1 = (1-v).*(1-u);
        c2 = (1-v).*u;
        c3 = v.*(1-u);
        c4 = v.*u;
        
        % Determine the indeces of the elements
        pa = vi  + ( ui  -1 )*no_el;
        pb = vi  + ( uin -1 )*no_el;
        pc = vin + ( ui  -1 )*no_el;
        pd = vin + ( uin -1 )*no_el;
        
        % Interpolate
        for n=1:no_element
            ndx = (element(n)-1)*no_az*no_el + [pa,pb,pc,pd];
            
            a = field_pattern_vertical( ndx );
            a = reshape( a,no_values,4 );
            
            V(:,n) = c1.*a(:,1) + c2.*a(:,2) + c3.*a(:,3) + c4.*a(:,4);
            
            a = field_pattern_horizontal( ndx );
            a = reshape( a,no_values,4 );
            
            H(:,n) = c1.*a(:,1) + c2.*a(:,2) + c3.*a(:,3) + c4.*a(:,4);
            
            if calc_cp
                a = common_phase( ndx );
                a = reshape( a,no_values,4 );
                
                CP(:,n) = c1.*a(:,1) + c2.*a(:,2) + c3.*a(:,3) + c4.*a(:,4);
            end
        end
        
    otherwise
        
        % Exapnd the interpolation grid to the left and right to reduce errors at
        % pi.
        
        a = numel(azimuth_grid);
        ind = [ a-5:a-1 1:a 2:5  ];
        agn = [ azimuth_grid(a-5:a-1)-2*pi azimuth_grid azimuth_grid(2:5)+2*pi ];
        
        % Call the MATLAB-internal interpolation functions
        for n = 1:no_element
            V(:,n) = interp2( agn , elevation_grid ,...
                field_pattern_vertical(:,ind,element(n)) , azimuth , elevation , h_array.interpolation_method );
            
            H(:,n) = interp2( agn , elevation_grid ,...
                field_pattern_horizontal(:,ind,element(n)) , azimuth , elevation , h_array.interpolation_method );
            
            if calc_cp
                CP(:,n) = interp2( agn , elevation_grid ,...
                    common_phase(:,ind,element(n)) , azimuth , elevation , h_array.interpolation_method );
            end
        end
end

% Remap output to match input dimensions

V = reshape( V, [dims,no_element] );
H = reshape( H, [dims,no_element] );

if calc_cp
    CP = reshape( atan2(imag(CP), real(CP)), [dims,no_element] );
end
 

% The effective distances are only needed when we calculate the channel
% coefficients. Therefore, it is also provided by the interpolation
% function.

if nargout > 3
    % Compute the effective distances for each path
    
    if any(any( element_position~=0 ))
        
        % Transform incoming angles from spherical coordinates to cartesian
        % coordinates.
        A = zeros(3,no_values);
        [A(1, :), A(2, :), A(3, :)] = sph2cart(azimuth, elevation, 1);
        
    end
    
    % The distances are calculated by a parallel projection.
    % See: http://de.wikipedia.org/wiki/Parallelprojektion
    % The normal vector of the pojection plane is given in X.
    % The original point P is the element position defined by the array
    % geometry. The origin of the projection plane is the origin of the
    % array coordinate system. The projection result is B.
    
    dist = zeros( no_values,no_element );
    for n=1:no_element
        E = element_position(:,n);
        
        if any( E ~= 0 )            % The distance is 0 if P is in the origin
            tmp = E'*A;             % cos(E,A)
            D = tmp([1 1 1],:) .* A;
            dist(:,n) = -sign( A(1,:).*D(1,:) + A(2,:).*D(2,:) + A(3,:).*D(3,:) ) .* ...
                sqrt(sum( D.^2 ));
            
            % This is the same as
            % for m = 1:no_values
            %    D = -( -E.' * A(:,m) ) * A(:,m) ;
            %    dist(m,n) = sign(D'*A(:,m)) * sqrt(sum(  D.^2 ));
            % end
        end
    end
    
    % Return the output
    dist = reshape( dist, [dims,no_element] );
end

end

