function generate( obj,array_type,element,p1,p2,p3 )
%GENERATE Generates predefined arrays
%
%   GENERATE( array_type ) generates a new, predefined antenna array given by
%   array_type. Currently supported are:
%
%   omni    An isotropic sphere radiating equally in both polarizations.
%   dipole  An ideal dipole radiating in vertical polarization.
%   xpol    Two elements with ideal isotropic patterns, one radiating in vertical
%           polarization and one radiating in horizontal polarization.
%   custom  An array with a custom set gain in elevation or azimuth.
%           E.g.: a.generate('custom',1,90,90,0.1);
%           create an array with 90 opening in azimuth and elevation and 10
%           dB front-back ratio.
%   rhcp-dipole, lhcp-dipole, rhcp-lhcp-dipole
%           Circular polarized antennas composed of dipoles
%   ulaX    Unified linear arrays composed of omni-antennas  with 10 cm
%           element distance. X can be 2,4 or 8.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

if nargin >= 2
    % If we have two input arguments, parse the type
    supported_types = array.supported_types;
    
    if ~( ischar(array_type) && any( strcmp(array_type,supported_types)) )
        str = 'Array type not found; supported types are: ';
        no = numel(supported_types);
        for n = 1:no
            str = [str,supported_types{n}];
            if n<no
                str = [str,', '];
            end
        end
        error(str);
    end
end

if nargin >= 3
    if ~( size(element,1) == 1 && isnumeric(element) && all(size(element) == [1 1]) ...
            &&  all( mod(element,1)==0 ) && min(element) >= 0 )
        error('??? "element" must be scalar, integer >= 0 and can not exceed array size')
    end
else
    element = 0;
end


if numel(obj) > 1
    % Do for parameter_set_arrays
    for n=1:numel(obj)
        obj(n).generate( array_type,1 );
    end
else
    if element > obj.no_elements
        obj.no_elements = element;
    end
    
    switch array_type
        case 'omni'
            if element == 0 || isempty(obj.no_elements)
                obj.no_elements                 = 1;
                obj.elevation_grid              = (-90:90)*pi/180;
                obj.azimuth_grid                = (-180:180)*pi/180;
                obj.element_position            = zeros( 3,1 );
                obj.field_pattern_vertical      = ones( 181,361 );
                obj.field_pattern_horizontal    = zeros( 181,361 );
                obj.pol_vector                  = [0;0;1];
                obj.coupling                    = 1;
            else
                obj.element_position(:,element)           = zeros( 3,1 );
                obj.field_pattern_vertical(:,:,element)   = ones( obj.no_el,obj.no_az );
                obj.field_pattern_horizontal(:,:,element) = zeros( obj.no_el,obj.no_az );
                obj.pol_vector(:,element)                 = [0;0;1];
            end
            
        case 'dipole'
            if element == 0 || isempty(obj.no_elements)
                obj.generate('omni');
                obj.interpolation_method        = 'linear';
                e = 1;
            else
                e = element;
            end
            
            tmp = sin( (1-1e-6)*obj.elevation_grid + pi/2 );
            P = zeros( obj.no_el,obj.no_az );
            for n = 1:numel( obj.azimuth_grid )
               P(:,n) = tmp;
            end
            P = P ./ sqrt( mean(mean(P.^2)) );
            obj.field_pattern_vertical(:,:,e) = P;
            obj.field_pattern_horizontal(:,:,e) = zeros( obj.no_el,obj.no_az );
            obj.pol_vector(:,e) = [0;0;1];
            
        case 'custom'
            if nargin < 4
                p1 = 120;
            end
            
            if nargin < 5
                p2 = 120;
            end
            
            if nargin < 6
                p3 = 0.1;
            end
            
            if ~( size(p1,1) == 1 && isnumeric(p1) && isreal(p1) &&...
                    all(size(p1) == [1 1]) )
                error('Azimuth FHWM (p1) has invalid value.')
            end
            
            if ~( size(p2,1) == 1 && isnumeric(p2) && isreal(p2) &&...
                    all(size(p2) == [1 1]) )
                error('Elevation FHWM (p2) has invalid value.')
            end
            
            if ~( size(p3,1) == 1 && isnumeric(p2) && isreal(p3) && ...
                    all(size(p3) == [1 1]) && p3>=0 && p3<0.5)
                error('Front-to-back ration (p3) has invalid value.')
            end
            
            if element == 0 || isempty(obj.no_elements)
                obj.generate('omni');
                obj.interpolation_method        = 'linear';
                e = 1;
            else
                e = element;
            end
            
            % Calculate the azimuth response
            ang = -180:180;
            ind = find(ang >=  p1/2,1);
            ang = ang*pi/180;
            
            a   = 1;        % Initial angle
            dm  = 0.5;      % Step size
            x   = inf;
            delta = Inf; ddir = +1; lp = 1;
            while lp<5000 && delta > 1e-7
                if lp>1; an = a + ddir * dm; delta = abs(a-an); else an = a; end
                
                C = p3+(1-p3)*exp(-an*ang.^2);
                xn = abs( C(ind)-0.5 );
                
                if xn < x; a = an; x = xn; else ddir = -ddir; dm = 0.382 * dm; end
                lp = lp + 1;
            end
            C = p3+(1-p3)*exp(-an*ang.^2);
            
            
            % Calculate the elevation response
            ang = -90:90;
            ind = find(ang >=  p2/2,1);
            ang = ang*pi/180;
            
            a   = 1;        % Initial angle
            dm  = 0.5;      % Step size
            x   = inf;
            delta = Inf; ddir = +1; lp = 1;
            while lp<5000 && delta > 1e-7
                if lp>1; an = a + ddir * dm; delta = abs(a-an); else an = a; end
                
                D = cos(ang).^an;
                xn = abs( D(ind)-0.5 );
                
                if xn < x; a = an; x = xn; else ddir = -ddir; dm = 0.382 * dm; end
                lp = lp + 1;
            end
            D = cos(ang).^an;
            
            
            P = zeros(181,361);
            for a = 1:181
                for b = 1:361
                    P(a,b) = D(a) * C(b);
                end
            end
            P(P<p3) = p3;
            P =  sqrt( P ./ mean(mean(P)) );
            
            obj.field_pattern_vertical(:,:,e) = P;
            obj.field_pattern_horizontal(:,:,e) = zeros( obj.no_el,obj.no_az );
            obj.pol_vector(:,e)       = [0;0;1];
            
        case 'xpol'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            obj.generate('omni');
            obj.copy_element(1,2);
            obj.rotate_pattern(90,'x',2);
            
        case 'rhcp-dipole'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            obj.generate('dipole');
            obj.generate('dipole',2);
            obj.rotate_pattern(90,'x',2);
            obj.coupling = 1/sqrt(2) * [1;1j];
            
        case 'lhcp-dipole'
            obj.generate('rhcp-dipole',element);
            obj.coupling = 1/sqrt(2) * [1;-1j];
            
        case 'rhcp-lhcp-dipole'
            obj.generate('rhcp-dipole',element);
            obj.coupling = 1/sqrt(2) * [1 1;1j -1j];
            
        case 'ula2'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            obj.generate('omni');
            obj.no_elements                 = 2;
            obj.element_position(2,:)       = [-0.05 0.05];
            
        case 'ula4'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            obj.generate('omni');
            obj.no_elements                 = 4;
            obj.element_position(2,:)       = -0.15 :0.1: 0.15;
            
        case 'ula8'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            obj.generate('omni');
            obj.no_elements                 = 8;
            obj.element_position(2,:)       = -0.35 :0.1: 0.35;
            
    end
    
    obj.name = array_type;
end
end

