function [ pol_vector, match ] = estimate_pol_vector( h_array , element , verbose  )
%ESTIMATE_POL_VECTOR Estimates the orientation vector from the patterns
%
%   [ pol_vector, match ] = ESTIMATE_POL_VECTOR( element , verbose )
%   This function estimates the orientation vector from the field patterns.
%   This allows the use of measured patterns in QuaDRiGa where the
%   orientation vector is unknown. Results are also stored in the
%   "pol_vector" property of the array object.
%
%   With the parameter "element", it is possible to select a specific
%   element of an array for which the evaluation should be done.
%
%   "verbose" can switch the progress bar on and off.
%
%   The output parameter "match" provides information on how well the
%   pol-vector matches the pattern. It scales from 0 to 1 where 1 is a
%   perfect match.
%
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


% Parse input variables.
if exist('element','var')
    if ~( size(element,1) == 1 && isnumeric(element) ...
            &&  all( mod(element,1)==0 ) && min(element) > 0 && max(element) ...
            <= h_array.no_elements )
        error('??? "element" must be integer > 0 and can not exceed array size')
    end
else
    element = 1: h_array.no_elements ;
end

if exist('verbose','var')
    if ~( isnumeric(verbose) || ~isnan(verbose) || all(size(verbose) == [1 1]) )
        error('"verbose" must be numeric.')
    else
        verbose = logical(verbose);
    end
else
    verbose = true;
end


% It is more efficient to run this calculation with a reduced array size.
if h_array.no_az == 361 && h_array.no_el == 181
    reduce_size = true;
else
    reduce_size = false;
end


% Show a progress bar.
if verbose
    fprintf('Orientation  [');
    vb_dots = 50;
    tStart = clock;
    m0=0;
    oo = 0;
    no_obj = 2*numel( element );
end

no_element = numel( element );

match = zeros(1,no_element);
for n = 1:no_element
    
    % Update progress bar
    if verbose;
        oo = oo + 1;
        m1=ceil(oo/no_obj*vb_dots); if m1>m0;
            for m2=1:m1-m0; fprintf('o'); end; m0=m1; end;
    end;
    
    % Copy current element from array
    ap = array('dipole');
    if reduce_size
        ap.field_pattern_vertical   = h_array.field_pattern_vertical(:,:,element(n));
        ap.field_pattern_horizontal = h_array.field_pattern_horizontal(:,:,element(n));
        ap.set_grid( (-180:5:180)*pi/180 , (-90:5:90)*pi/180 );
    else
        ap.elevation_grid = h_array.elevation_grid;
        ap.azimuth_grid = h_array.azimuth_grid;
        ap.field_pattern_vertical   = h_array.field_pattern_vertical(:,:,element(n));
        ap.field_pattern_horizontal = h_array.field_pattern_horizontal(:,:,element(n));
    end
    
    % The power normalization
    N = abs(ap.field_pattern_vertical).^2 + abs(ap.field_pattern_horizontal).^2;
    N = sqrt( mean( reshape(N,[],1) ) );
    
    ap.field_pattern_vertical = ap.field_pattern_vertical / N;
    ap.field_pattern_horizontal = ap.field_pattern_horizontal / N;
    
    apV = abs(ap.field_pattern_vertical).^2;
    apH = abs(ap.field_pattern_horizontal).^2;
    
    
    % Low-resolution global search
    % This is needed since there might be several local minima of the cost
    % function. The global search tries to find start values that are close
    % to the global minima. Then, we use the SAGE algorithm to find the
    % exact value of this minima.
    
    az = -175:25:175;
    el = -80:20:80;
    
    no_el = numel(el);
    no_az = numel(az);
    
    C = zeros( no_el , no_az );
    for a = 1:no_az
        for e = 1:no_el
            
            b = ap.copy_objects;
            [xx,yy,zz] = sph2cart( az(a)*pi/180 , el(e)*pi/180 , 1);
            b.pol_vector = [xx,yy,zz]';
            b.rotate_pattern( 0 , 'x' );
            
            % The cost function
            C(e,a) = sum(sum(  abs(apV - abs(b.field_pattern_vertical).^2) + ...
                abs(apH - abs(b.field_pattern_horizontal).^2)  ));
        end
    end
    
    % For debug:
    % imagesc( az,el,10*log10(C) );colorbar
    
    % Update progress bar
    if verbose;
        oo = oo + 1;
        m1=ceil(oo/no_obj*vb_dots); if m1>m0;
            for m2=1:m1-m0; fprintf('o'); end; m0=m1; end;
    end;
    
    % Set the initial values
    
    [~,ind] = min(reshape( C , [] , 1 ));
    [e,a] = ind2sub(size(C),ind);
    
    az = az(a);             % Initial azimuth angle
    el = el(e);             % Initial elevation angle
    
    % Use the SAGE algorithm to find to orientation vector.
    
    % Optimize azimuth angle
    a   = az;               % Initial angle
    dm  = 10;               % Step size in degree
    x   = inf;
    delta = Inf; ddir = +1; lp = 1;
    while lp<100 && delta > 1e-2
        if lp>1; an = a + ddir * dm; delta = abs(a-an); else an = a; end
        
        % Apply the pol_vector
        b = ap.copy_objects;
        [xx,yy,zz] = sph2cart( an*pi/180 , el*pi/180 , 1);
        b.pol_vector = [xx,yy,zz]';
        b.rotate_pattern( 0 , 'x' );
        
        % The cost function
        xn = sum(sum(  abs(apV - abs(b.field_pattern_vertical).^2) + ...
            abs(apH - abs(b.field_pattern_horizontal).^2)  ));
        
        if xn < x; a = an; x = xn; else ddir = -ddir; dm = 0.382 * dm; end
        lp = lp + 1;
    end
    az = an;
    
    % Optimize elevation angle
    a   = el;      % Initial angle
    dm  = 10;      % Step size in degree
    x   = inf;
    delta = Inf; ddir = +1; lp = 1;
    while lp<5000 && delta > 1e-2
        if lp>1; an = a + ddir * dm; delta = abs(a-an); else an = a; end
        
        % Apply the pol_vector
        b = ap.copy_objects;
        [xx,yy,zz] = sph2cart( az*pi/180 , an*pi/180 , 1);
        b.pol_vector = [xx,yy,zz]';
        b.rotate_pattern( 0 , 'x' );
        
        % The cost function
        xn = sum(sum(  abs(apV - abs(b.field_pattern_vertical).^2) + ...
            abs(apH - abs(b.field_pattern_horizontal).^2)  ));
        
        if xn < x; a = an; x = xn; else ddir = -ddir; dm = 0.382 * dm; end
        lp = lp + 1;
    end
    el = an;
    
    match(n) = 1 - xn/(2*ap.no_az*ap.no_el) ;
    
    % There might be a remaining sign error since we only considered the
    % amplitude of the pattern, but not the phase. Here, we catch wrong
    % signs and write the estimated orientation vector to the array object.
    
    % Determine the sign
    tmp = angle(ap.field_pattern_vertical) - angle(b.field_pattern_vertical);
    A = abs( angle( exp( 1j*( tmp ) ) ) );
    
    tmp = angle(ap.field_pattern_horizontal) - angle(b.field_pattern_horizontal);
    B = abs( angle( exp( 1j*( tmp ) ) ) );
    
    % Write the orientation vector to array
    [xx,yy,zz] = sph2cart( az*pi/180 , el*pi/180 , 1);
    if sum( reshape( [A,B] ,[],1)>(pi/2) ) / (2*numel(A)) > 0.5
        h_array.pol_vector(:,element(n)) = -[xx,yy,zz]';
    else
        h_array.pol_vector(:,element(n)) = [xx,yy,zz]';
    end
end

% Return argument
pol_vector = h_array.pol_vector(:,element);

% Update progress bar and show time
if verbose
    fprintf('] %5.0f seconds\n',round( etime(clock, tStart) ));
end

end

