function [ common_phase , match ] = estimate_common_phase( h_array , element )
%ESTIMATE_COMMON_PHASE Estimetes the common pahse from the field patterns
%
%   It is possible that antenna patterns have a phase component. For
%   example, an antenna array might be assembled out of several elements,
%   which are at different position in the array. When the patterns of the
%   elements are then calibrated in the lab, the individual positions
%   result in a phase offset which is part of the measured pattern
%   response. The core function of QuaDRiGa, however, only uses the real
%   part if the provided patterns. Hence, not calibrating the phase offset
%   out of the pattern will lead to errors.       
% 
%   [ common_phase , match ] = ESTIMATE_COMMON_PHASE( element )
%   This function calculates the phase from the pattern and stores it in
%   the "common_phase" property of the antenna array object. However, this
%   requires that the orientation vectors are set correctly, i.e. you need
%   to call "estimate_pol_vector" first when importing measured patterns.   
%
%   With the parameter "element", it is possible to select a specific
%   element of an array for which the evaluation should be done.
%
%   "verbose" can switch the progress bar on and off.
%
%   The output parameter "match" provides information on how well the
%   pol-vector matches the pattern. It scales from 0 to 1 where 1 is a
%   perfect match.
%
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


% Parse input variables.
if exist('element','var')
    if ~( size(element,1) == 1 && isnumeric(element) ...
            &&  all( mod(element,1)==0 ) && min(element) > 0 && max(element) ...
            <= h_array.no_elements )
        error('??? "element" must be integer > 0 and can not exceed array size')
    end
else
    element = 1:h_array.no_elements ;
end

no_element = numel(element);


% For calculating the match
if nargout == 2
    N = abs(h_array.field_pattern_vertical(:,:,element)).^2 +...
        abs(h_array.field_pattern_horizontal(:,:,element)).^2;
    N = sqrt( mean( reshape(N,[],no_element) ) );
    
    apV = zeros( h_array.no_el , h_array.no_az , no_element );
    apH = zeros( h_array.no_el , h_array.no_az , no_element );
    for n = 1:no_element
        apV(:,:,n) = abs(h_array.field_pattern_vertical(:,:,element(n)) /N(n) ).^2;
        apH(:,:,n) = abs(h_array.field_pattern_horizontal(:,:,element(n)) /N(n) ).^2;
    end
end


% This calculates the common phase and removes the phase information from
% the field patterns.

cp_old = h_array.common_phase(:,:,element);
cp = h_array.rotate_pattern( 0 , 'y' , element , 2 );
common_phase = angle( exp( 1j*(cp_old + cp)) );
h_array.common_phase(:,:,element) = common_phase;

% Calculate the match
if nargout == 2
    match = zeros(1,no_element);
    for n = 1:numel(element)
        match(n) = sum(sum(  abs(apV(:,:,n) -...
            abs(h_array.field_pattern_vertical(:,:,element(n))/N(n)).^2) + ...
            abs(apH(:,:,n) -...
            abs(h_array.field_pattern_horizontal(:,:,element(n))/N(n)).^2)  ));
    end
    match = 1 - match/(2 * h_array.no_az * h_array.no_el) ;
end

end


