classdef array < handle & matlab.mixin.Copyable
%ARRAY Antenna array class
%
% DESCRIPTION
% This class combines all functions to create and edit antenna arrays. An antenna
% array is a set of single antenna elements, each having a specific beam pattern,
% that can be combined in any geometric arrangement. A set of synthetic arrays
% that allow simulations without providing your own antenna patterns is
% provided (see generate method for more details).
%
% REFERENCE
% The main functionality was taken from the Winner channel model. "Kyösti, P.;
% Meinilä, J.; Hentilä, L. & others; {IST-4-027756 WINNER II D1.1.2 v.1.1}:
% WINNER II Channel Models; 2007". New functionality has been added to provide
% geometric polarization calculations and antenna pattern rotations.
%
% EXAMPLE
% This example creates an array of crossed dipoles.
%
%    a = array;                           % Create new array object
%    a.generate('dipole');                % Generate a synthetic dipole pattern
%    a.copy_element(1,2);                 % Duplicate the dipole
%    a.rotate_pattern(90,'y',2);          % Rotate the second element by 90°
%    a.visualize;                         % Show the output
%
%
% array Properties:
%    name - Name of the antenna array
%    interpolation_method - Method for interpolating the beam pattern
%    no_elements - Number of antenna elements
%    elevation_grid - Positions, where the elevation grid is defined
%    azimuth_grid - Positions, where the azimuth grid is defined
%    element_position - Physical position of the element (in local cartesian coordinates)
%    field_pattern_vertical - Vertical polarized field pattern
%    field_pattern_horizontal - Horizontal polarized field pattern
%    common_phase - A phase component resulting from different element positions
%    pol_vector - Polarization vector
%    coupling - Coupling between elements
%    no_az - Number of azimuth values
%    no_el - Number of elevation values
%
% array Methods:
%    copy_element - Creates a copy of an antenna element
%    copy_objects - Creates a deep copy of the current array object
%    estimate_pol_vector - Estimates the orientation vector from the patterns
%    generate - Generates predefined arrays
%    interpolate - Interpolates the field pattern
%    rotate_pattern - Rotates antenna patterns
%    set_grid - Sets a new grid for azimuth and elevation
%    visualize - Create a plot showing the element configurations
%    winner_compatibe_output - Create Winner-compatible output data structures
%
% Additional Comments:
%    - Increasing the number of elements creates new elements which are
%      initialized as copies of the first element.
%    - Decreasing the number of elements deletes the last elements from the
%      array.
%
%
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% Fraunhofer Heinrich Hertz Institute
% Wireless Communication and Networks
% Einsteinufer 37, 10587 Berlin, Germany
%  
% This file is part of QuaDRiGa.
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% QuaDRiGa is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%     
% You should have received a copy of the GNU Lesser General Public License
% along with QuaDRiGa. If not, see <http://www.gnu.org/licenses/>.
    
    properties
        name = 'New array';                     % Name of the antenna array
        interpolation_method = 'linear';        % Method for interpolating the beam pattern
    end
    
    properties(Dependent)
        no_elements                             % Number of antenna elements
        elevation_grid                          % Positions, where the elevation grid is defined
        azimuth_grid                            % Positions, where the azimuth grid is defined
        element_position                        % Physical position of the element (in local cartesian coordinates)
        field_pattern_vertical                  % Vertical polarized field pattern
        field_pattern_horizontal                % Horizontal polarized field pattern
        common_phase                            % A phase component resulting from different element positions
        pol_vector                              % Polarization vector
        coupling                                % Coupling between elements
    end
    
    properties(SetAccess=private)
        no_az = 5;                              % Number of azimuth values
        no_el = 3;                              % Number of elevation values
    end

    properties(Access=private)
        Pno_elements                = 1;
        Pelevation_grid             = [ -1.570796326794897,0,1.570796326794897];
        Pazimuth_grid               = [ -3.141592653589793,-1.570796326794897,0,...
            1.570796326794897,3.141592653589793];
        Pelement_position           = [0;0;0];
        Pfield_pattern_vertical     = ones(3,5);
        Pfield_pattern_horizontal   = zeros(3,5);
        Pcommon_phase               = zeros(3,5);
        Ppol_vector                 = [0;0;1];
        Pcoupling                   = 1;
    end
    
    methods
        % The constructor
        function obj = array(array_type,p1,p2,p3)
            if nargin==1
                obj.generate(array_type);
            elseif nargin == 2
                obj.generate(array_type,0,p1);
            elseif nargin == 3
                obj.generate(array_type,0,p1,p2);
              elseif nargin == 4
                obj.generate(array_type,0,p1,p2,p3);  
            end
        end
        
        % Get functions
        function out = get.no_elements(obj)
            out = obj.Pno_elements;
        end
        function out = get.elevation_grid(obj)
            out = obj.Pelevation_grid;
        end
        function out = get.azimuth_grid(obj)
            out = obj.Pazimuth_grid;
        end
        function out = get.element_position(obj)
            out = obj.Pelement_position;
        end
        function out = get.field_pattern_vertical(obj)
            out = obj.Pfield_pattern_vertical;
        end
        function out = get.field_pattern_horizontal(obj)
            out = obj.Pfield_pattern_horizontal;
        end
        function out = get.common_phase(obj)
            out = obj.Pcommon_phase;
        end
        function out = get.pol_vector(obj)
            out = obj.Ppol_vector;
        end
        function out = get.coupling(obj)
            out = obj.Pcoupling;
        end
        
        % Set functions
        function set.name(obj,value)
            if ~( ischar(value) )
                error('QuaDRiGa:Array:wrongInputValue','??? "name" must be a string.')
            end
            obj.name = value;
        end
        
        function set.interpolation_method(obj,value)
            supported_types = {'nearest','linear','spline','linear_int','nearest_int'};
            if ~( ischar(value) && any( strcmpi(value,supported_types)) )
                str = 'Interpolation method not supported; supported types are: ';
                no = numel(supported_types);
                for n = 1:no
                    str = [str,supported_types{n}];
                    if n<no
                        str = [str,', '];
                    end
                end
                error('QuaDRiGa:Array:wrongInterpolationType',str);
            end
            obj.interpolation_method = value;
        end
        
        function set.no_elements(obj,value)
            if ~( all(size(value) == [1 1]) && isnumeric(value) ...
                    && isreal(value) && mod(value,1)==0 && value > 0 )
                error('QuaDRiGa:Array:wrongInputValue','??? "no_elements" must be integer and > 0')
            end
            
            if obj.no_elements > value
                obj.Pelement_position           = obj.Pelement_position(:,1:value);
                obj.Pfield_pattern_vertical     = obj.Pfield_pattern_vertical(:,:,1:value);
                obj.Pfield_pattern_horizontal   = obj.Pfield_pattern_horizontal(:,:,1:value);
                obj.Pcommon_phase               = obj.Pcommon_phase(:,:,1:value);
                obj.Ppol_vector                 = obj.Ppol_vector(:,1:value);
                
                ne = obj.no_elements-value;
                nc = size(obj.Pcoupling);
                obj.Pcoupling = obj.Pcoupling( 1:nc(1)-ne , 1:max(nc(2)-ne,1) );
                
            elseif obj.no_elements < value
                ne = value-obj.no_elements;
                
                obj.Pelement_position = [ obj.Pelement_position,...
                    obj.Pelement_position(:,ones( 1,ne )) ];
                
                obj.Ppol_vector       = [ obj.Ppol_vector,...
                    obj.Ppol_vector(:,ones( 1,ne )) ];
                
                obj.Pfield_pattern_vertical = cat( 3, obj.Pfield_pattern_vertical ,...
                    obj.Pfield_pattern_vertical(:,:,ones( 1,ne )));
                
                obj.Pfield_pattern_horizontal = cat( 3, obj.Pfield_pattern_horizontal ,...
                    obj.Pfield_pattern_horizontal(:,:,ones( 1,ne )));
                
                obj.Pcommon_phase = cat( 3, obj.Pcommon_phase ,...
                    obj.Pcommon_phase(:,:,ones( 1,ne )));
                
                nc = size(obj.Pcoupling);
                C = zeros( nc(1)+ne , nc(2)+ne);
                for n = 1:ne
                   C( nc(1)+n,nc(2)+n ) = 1; 
                end
                C( 1:nc(1) , 1:nc(2) ) = obj.Pcoupling;
                obj.Pcoupling = C;
            end
            
            obj.Pno_elements = value;
        end
        
        function set.elevation_grid(obj,value)
            if ~( any( size(value) == 1 ) && isnumeric(value) && isreal(value) &&...
                    max(value)<=pi/2 && min(value)>=-pi/2 )
                error('QuaDRiGa:Array:wrongInputValue','??? "elevation_grid" must be a vector containing values between -pi/2 and pi/2')
            end
            
            val_old = size(obj.field_pattern_vertical,1);
            val_new = numel(value);
            if val_old > val_new
                obj.Pfield_pattern_horizontal   = obj.Pfield_pattern_horizontal( 1:val_new ,: , : );
                obj.Pfield_pattern_vertical     = obj.Pfield_pattern_vertical( 1:val_new ,: , : );
                obj.Pcommon_phase               = obj.Pcommon_phase( 1:val_new ,: , : );
                
            elseif val_old < val_new
                a = size( obj.field_pattern_vertical );
                if numel(a) == 2
                    a(3) = 1;
                end
                b = val_new-val_old;
                
                obj.Pfield_pattern_horizontal = cat( 1, obj.Pfield_pattern_horizontal ,...
                    ones( b ,a(2),a(3))  );
                
                obj.Pfield_pattern_vertical = cat( 1, obj.Pfield_pattern_vertical ,...
                    ones( b ,a(2),a(3))  );
                
                obj.Pcommon_phase = cat( 1, obj.Pcommon_phase ,...
                    zeros( b ,a(2),a(3))  );
            end
            
            if size(value,1) ~= 1
                obj.Pelevation_grid = value';
            else
                obj.Pelevation_grid = value;
            end
            obj.no_el = val_new;
        end
        
        function set.azimuth_grid(obj,value)
            if ~( any( size(value) == 1 ) && isnumeric(value) && isreal(value) &&...
                    max(value)<=pi && min(value)>=-pi )
                error('QuaDRiGa:Array:wrongInputValue','??? "azimuth_grid" must be a vector containing values between -pi and pi')
            end
            
            val_old = size(obj.field_pattern_vertical,2);
            val_new = numel(value);
            if val_old > val_new
                obj.Pfield_pattern_horizontal   = obj.Pfield_pattern_horizontal( : , 1:val_new , : );
                obj.Pfield_pattern_vertical     = obj.Pfield_pattern_vertical( : , 1:val_new  , : );
                obj.Pcommon_phase               = obj.Pcommon_phase( : , 1:val_new  , : );
                
            elseif val_old < val_new
                a = size( obj.field_pattern_vertical );
                if numel(a) == 2
                    a(3) = 1;
                end
                b = val_new-val_old;
                
                obj.Pfield_pattern_horizontal = cat( 2, obj.Pfield_pattern_horizontal ,...
                    ones( a(1) , b , a(3))  );
                
                obj.Pfield_pattern_vertical = cat( 2, obj.Pfield_pattern_vertical ,...
                    ones( a(1) , b , a(3))  );
                
                obj.Pcommon_phase = cat( 2, obj.Pcommon_phase ,...
                    zeros( a(1) , b , a(3))  );
            end
            
            if size(value,1) ~= 1
                obj.Pazimuth_grid = value';
            else
                obj.Pazimuth_grid = value;
            end
            obj.no_az = val_new;
        end
        
        function set.element_position(obj,value)
            if ~( isnumeric(value) && isreal(value) )
                error('QuaDRiGa:Array:wrongInputValue','??? "element_position" must consist of real numbers')
            elseif ~all( size(value,1) == 3 )
                error('QuaDRiGa:Array:wrongInputValue','??? "element_position" must have 3 rows')
            end
            if size(value,2) ~= obj.no_elements
                obj.no_elements = size(value,2);
            end
            obj.Pelement_position = value;
        end
        
        function set.field_pattern_vertical(obj,value)
            a = numel( obj.Pelevation_grid );
            b = numel( obj.Pazimuth_grid );
            
            if obj.Pno_elements == 1
                dims = [ a , b ];
            else
                dims = [ a , b , obj.Pno_elements];
            end
            
            if ~( isnumeric(value) )
                error('QuaDRiGa:Array:wrongInputValue','??? "field_pattern_vertical" must be numeric.')
            elseif ~( numel(size(value)) == numel(dims) && all( size(value) == dims ) )
                error('QuaDRiGa:Array:wrongInputValue',['??? "field_pattern_vertical" must be of size [',num2str(a),'x',num2str(b),...
                    'x',num2str(obj.Pno_elements),'].'])
            end
            obj.Pfield_pattern_vertical = value;
        end
        
        function set.field_pattern_horizontal(obj,value)
            a = numel( obj.Pelevation_grid );
            b = numel( obj.Pazimuth_grid );
            
            if obj.no_elements == 1
                dims = [ a , b ];
            else
                dims = [ a , b , obj.Pno_elements];
            end
            
            if ~( isnumeric(value) )
                error('QuaDRiGa:Array:wrongInputValue','??? "field_pattern_horizontal" must be numeric.')
            elseif ~( numel( size(value) ) == numel( dims ) && all( size(value) == dims ) )
                error('QuaDRiGa:Array:wrongInputValue',['??? "field_pattern_horizontal" must be of size [',num2str(a),'x',num2str(b),...
                    'x',num2str(obj.Pno_elements),'].'])
            end
            obj.Pfield_pattern_horizontal = value;
        end
        
        function set.common_phase(obj,value)
            a = numel( obj.Pelevation_grid );
            b = numel( obj.Pazimuth_grid );
            
            if obj.no_elements == 1
                dims = [ a , b ];
            else
                dims = [ a , b , obj.Pno_elements];
            end
            
            if ~( isnumeric(value) )
                error('QuaDRiGa:Array:wrongInputValue','??? "field_pattern_horizontal" must be numeric.')
            elseif ~( numel( size(value) ) == numel( dims ) && all( size(value) == dims ) )
                error('QuaDRiGa:Array:wrongInputValue',['??? "field_pattern_horizontal" must be of size [',num2str(a),'x',num2str(b),...
                    'x',num2str(obj.Pno_elements),'].'])
            end
            obj.Pcommon_phase = value;
        end
        
        function set.pol_vector(obj,value)
            if ~( isnumeric(value) && isreal(value) )
                error('QuaDRiGa:Array:wrongInputValue','??? "pol_vector" must consist of real numbers')
            elseif ~all( size(value) == [3,obj.Pno_elements] )
                error('QuaDRiGa:Array:wrongInputValue','??? "pol_vector" must have 3 rows and the number of columns must match "no_elements"')
            end
            obj.Ppol_vector = value;
        end
       
        function set.coupling(obj,value)
            if ~( isnumeric(value) && size(value,1) == obj.Pno_elements && ...
                    size(value,2) >= 1 && size(value,2) <=  obj.Pno_elements )
                error('QuaDRiGa:Array:wrongInputValue','??? "coupling" must be a matrix with rows equal to elements and columns equal to ports')
            end
            obj.Pcoupling = value;
        end
    end

    methods(Static)
        function types = supported_types
            types =  {'omni','dipole','custom',...
                'rhcp-dipole','lhcp-dipole','rhcp-lhcp-dipole',...
                'xpol','ula2','ula4','ula8'};
        end
    end
end
