classdef simulation_parameters < handle & matlab.mixin.Copyable
%SIMULATION_PARAMETERS General configuration settings
%
% DESCRIPTION
% This class controls the simulation options and calculates constants for other
% classes. Currently, the following options can be set:
%
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% Fraunhofer Heinrich Hertz Institute
% Wireless Communication and Networks
% Einsteinufer 37, 10587 Berlin, Germany
%
% This file is part of QuaDRiGa.
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% QuaDRiGa is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%
% You should have received a copy of the GNU Lesser General Public License
% along with QuaDRiGa. If not, see <http://www.gnu.org/licenses/>. 
    
    properties(Dependent)
        
        % The number of samples per half-wave length
        % 	Sampling density describes the number of samples per half-wave
        % 	length. To fulfill the sampling theorem, the minimum sample
        % 	density must be 2. For  smaller values, interpolation of the
        % 	channel for variable speed is not  possible. On the other hand,
        % 	high values significantly increase the computing time
        % 	significantly. A good value is around 4. 
        sample_density                              
        samples_per_meter                           % Samples per one meter
        
        % Precision of the drifting functionality
        %   drifting_precision = 0
        %   This method applies rotating phasors to each path which emulates time
        %   varying Doppler characteristics. However, the large-scale parameters
        %   (departure and arrival angles, shadow fading, delays, etc.) are not
        %   updated in this case. This mode requires the least computing resources
        %   and may be preferred when only short linear tracks (up to several cm) are
        %   considered and the distance between transmitter and receiver is large.
        %
        %   drifting_precision = 1 (default)
        %   When drifting is enabled, all LSPs are updated for each snapshot. This
        %   requires significantly more computing resources but also increases the
        %   accuracy of the results. Drifting is required when e.g. non-linear tracks
        %   are generated or the distance between transmitter and receiver is small
        %   (below 20 m).
        %
        %   drifting_precision = 2
        %   LSPs are updated for each snapshot and for each antenna element at the
        %   receiver. This increases the accuracy for multi-element antenna arrays.
        %
        %   drifting_precision = 3
        %   This option also calculates the shadow fading, path loss and K-factor for
        %   each antenna separately. This feature tends to predict higher capacities
        %   since is also increases the randomness of the power for different MIMO
        %   elements. Use with care.
        drifting_precision
        
    end
    
    properties
        %  Update drifting paths when aoa changes [degrees]
        %    Updating the polarization rotation for large antenna arrays is
        %    very computing intensive and may slow down the simulations
        %    significantly. This parameter reduces the update rate. A value
        %    of 0.2 degrees (Default Setting) means that the polarization
        %    rotation is only updated when any of the four angles (AoA,
        %    EoA, AoD, EoD) changes more than 0.2 degrees. Otherwise it is
        %    kept constant at its last value. For highest precision
        %    calculations, set drifting_update_threshold to 0.       
        drifting_update_threshold   = 0.2;
        
        % Select the polarization rotation method
        %   use_polarization_rotation = 0
        %   Uses the polarization method from WINNER. No polarization
        %   rotation is calculated.
        %
        %   use_polarization_rotation = 1
        %   Uses the new polarization rotation method where the XPR is
        %   modeled by a rotation matrix. No change of circular
        %   polarization is assumed. 
        %
        %   use_polarization_rotation = 2 [Default]
        %   Uses the polarization rotation with an additional phase offset
        %   between the H and V component of the NLOS paths. The offset
        %   angle is calculated to match the XPR for circular polarization.
        %
        %   use_polarization_rotation = 3
        %   Uses polarization rotation for the geometric polarization but
        %   models the NLOS polarization change as in WINNER.
        use_polarization_rotation   = 2;            
        
        % Returns absolute delays in channel impulse response
        %   By default, delays are calculated such that the LOS delay is
        %   normalized to 0. By setting use_absolute_delays to 1 or true,
        %   the absolute path delays are included in channel.delays at the
        %   output of the model.   
        use_absolute_delays         = false;
        
        % Return all 20 subpaths
        %   By default, the complex valued amplitudes of 20 sub-paths are
        %   summed up to calculate the path amplitude. Setting
        %   use_subpath_output to 1 returns the individual amplitudes of
        %   each subpath in channel.coeff at the output of the model.
        use_subpath_output          = false;       
        
        show_progress_bars          = true;         % Show a progress bar on the MATLAB prompt
        center_frequency            = 2.6e9;        % Center frequency in [Hz]
        map_resolution              = 1;            % Resolution of the decorrelation maps in [samples/m]
    end
    
    properties(Constant)
        version = '1.2.3-307';                      % Version number of the current QuaDRiGa release (constant)
        speed_of_light = 299792458;                 % Speed of light (constant)
    end
    
    properties(Dependent,SetAccess=protected)
        wavelength                                  % Carrier wavelength in [m] (read only)
    end
    
    properties(Access=private)
        Psample_density             = 2.5;
        Pdrifting_precision         = 1;
    end
    
    methods
        
        % Get functions
        function out = get.sample_density(obj)
            out = obj.Psample_density;
        end
        function out = get.samples_per_meter(obj)
            out = 2*obj.center_frequency*obj.Psample_density / obj.speed_of_light;
        end
        function out = get.wavelength(obj)
            out = obj.speed_of_light / obj.center_frequency;
        end
        function out = get.drifting_precision(obj)
            out = obj.Pdrifting_precision;
        end
        
        % Set functions
        function set.sample_density(obj,value)
            if ~( all(size(value) == [1 1]) && isnumeric(value) && isreal(value) && value > 0 )
                error('??? Invalid sample density. The value must be real and > 0.')
            end
            obj.Psample_density = value;
        end
        
        function set.samples_per_meter(obj,value)
            if ~( all(size(value) == [1 1]) && isnumeric(value) && isreal(value) && value > 0 )
                error('??? Invalid samples_per_meter. The value must be real and > 0.')
            end
            obj.Psample_density = value*obj.wavelength/2;
        end
        
        function set.center_frequency(obj,value)
            if ~( all(size(value) == [1 1]) && isnumeric(value) && isreal(value) && value >= 0 )
                error('??? Invalid center frequency. The value must be real and > 0.')
            end
            obj.center_frequency = value;
        end
        
        function set.use_absolute_delays(obj,value)
            if ~( all(size(value) == [1 1]) ...
                    && (isnumeric(value) || islogical(value)) ...
                    && any( value == [0 1] ) )
                error('??? "use_absolute_delays" must be 0 or 1')
            end
            obj.use_absolute_delays = logical( value );
        end
        
        function set.drifting_precision(obj,value)
            if ~( all(size(value) == [1 1]) ...
                    && isnumeric(value) ...
                    && any( value == [0,1,2,3] ) )
                error('??? "drifting_precision" must be 0, 1, 2 or 3. Type ''help simulation_parameters.drifting_precision'' for more infos.')
            end
            obj.Pdrifting_precision = value;
        end
        
        function set.use_polarization_rotation(obj,value)
            if ~( all(size(value) == [1 1]) ...
                    && (isnumeric(value) || islogical(value)) ...
                    && any( value == 0:3 ) )
                error('??? "use_polarization_rotation" must be 0, 1, 2 or 3')
            end
            obj.use_polarization_rotation = value;
        end
        
        function set.use_subpath_output(obj,value)
            if ~( all(size(value) == [1 1]) ...
                    && (isnumeric(value) || islogical(value)) ...
                    && any( value == [0 1] ) )
                error('??? "use_subpath_output" must be 0 or 1')
            end
            obj.use_subpath_output = logical( value );
        end
        
        function set.map_resolution(obj,value)
            if ~( all(size(value) == [1 1]) && isnumeric(value) ...
                    && isreal(value) && value > 0 )
                error('??? "map_resolution" must be real, scalar and > 0')
            end
            obj.map_resolution = value;
        end
        
        function set.show_progress_bars(obj,value)
            if ~( all(size(value) == [1 1]) ...
                    && (isnumeric(value) || islogical(value)) ...
                    && any( value == [0 1] ) )
                error('??? "use_subpath_output" must be 0 or 1')
            end
            obj.show_progress_bars = logical( value );
        end
    end
end

