function h_parset = generate_correlation_maps( h_parset, initialize, vb_dots )
%GENERATE_CORRELATION_MAPS Generates a new set of correlation maps
%
%   GENERATE_CORRELATION_MAPS manually generates the correlation maps.
%   Correlation maps are needed to handle the distance-dependant correlation
%   between mobile terminals. E.g. when two terminals are close to each other,
%   they will see similar channels.
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input parameters
if exist( 'initialize' , 'var' ) && ~isempty( initialize )
    if ~( all(size(initialize) == [1 1]) ...
            && (isnumeric(initialize) || islogical(initialize)) ...
            && any( initialize == [0 1] ) )
        error('??? "initialize" must be 0 or 1')
    end
else
    initialize = true;
end

verbose = h_parset(1).simpar.show_progress_bars;
if nargin < 3
    if verbose
        vb_dots = ones(1,numel(h_parset))*floor( 50/numel(h_parset) );
        tmpA = numel( vb_dots );
        tmpB = 50-sum( vb_dots );
        tmpC = 1 : floor( tmpA/tmpB ) : tmpA;
        for n = 1:tmpB
            vb_dots( tmpC(n) ) = vb_dots( tmpC(n) )+1;
        end
    else
         vb_dots = zeros(1,numel(h_parset));
    end
end

if nargin < 3 && verbose && initialize
    fprintf('Parameters   [');
    tStart = clock;
end

if numel(h_parset) > 1
    for n=1:numel(h_parset)
        generate_correlation_maps( h_parset(n), initialize, vb_dots(n) );
    end
    
elseif h_parset.no_positions > 0
    
    if ~h_parset.LSP_matrix_isOK
        error('LSP_matrix is not positive-definite.');
    end
    
    % Get the autocorrelation distances and scale them to the map resolution.
    delta = [ h_parset.scenpar.DS_lambda,...
        h_parset.scenpar.KF_lambda,...
        h_parset.scenpar.SF_lambda,...
        h_parset.scenpar.AS_D_lambda,...
        h_parset.scenpar.AS_A_lambda,...
        h_parset.scenpar.ES_D_lambda,...
        h_parset.scenpar.ES_A_lambda] .* h_parset.samples_per_meter;

    % Extract the minimum and maximum positions from the list of positions
    min_pos = floor( min( h_parset.positions-h_parset.map_extension,[],2 )...
        * h_parset.samples_per_meter)/h_parset.samples_per_meter;
    max_pos = ceil ( max( h_parset.positions+h_parset.map_extension,[],2 )...
        * h_parset.samples_per_meter)/h_parset.samples_per_meter;
    
    % Determine the map edges taking an existing grid into account
    x_min = min( [ h_parset.map_extent(1, 1) , min_pos(1) ] );
    x_max = max( [ h_parset.map_extent(1, 2) , max_pos(1) ] );
    y_min = min( [ h_parset.map_extent(2, 1) , min_pos(2) ] );
    y_max = max( [ h_parset.map_extent(2, 2) , max_pos(2) ] );
    
    % Extending the map while keeping the sampling rate constant
    h_parset.map_extent = [x_min, x_max; y_min, y_max];
    
    % Calculate maps by filtering of random sequences
    no_map_y = h_parset.map_size(2);
    no_map_x = h_parset.map_size(1);

    if initialize
        h_parset.Pparameter_maps = zeros( no_map_y , no_map_x , 7 );
        h_parset = generate_map_filter( h_parset , delta , vb_dots );
        
        % Generate cross-correlation of the maps by a linear transformation
        R_sqrt = sqrtm( h_parset.LSP_xcorr_matrix );
        tmp = reshape( h_parset.Pparameter_maps , no_map_y*no_map_x , 7 ) * R_sqrt;
        h_parset.Pparameter_maps = reshape( tmp , no_map_y , no_map_x , 7 );
        
        % Transform Normal distributed maps to scenario specific distributions
        a = h_parset.scenpar.DS_mu;
        b = h_parset.scenpar.DS_sigma;
        c = h_parset.scenpar.AS_D_mu;
        d = h_parset.scenpar.AS_D_sigma;
        e = h_parset.scenpar.AS_A_mu;
        f = h_parset.scenpar.AS_A_sigma;
        g = h_parset.scenpar.SF_sigma;
        h = h_parset.scenpar.KF_mu;
        k = h_parset.scenpar.KF_sigma;
        l = h_parset.scenpar.ES_D_mu;
        m = h_parset.scenpar.ES_D_sigma;
        n = h_parset.scenpar.ES_A_mu;
        o = h_parset.scenpar.ES_A_sigma;
        
        h_parset.Pparameter_maps(:,:,1) = b*h_parset.Pparameter_maps(:,:,1) + a;   % Delay Spread
        h_parset.Pparameter_maps(:,:,2) = k*h_parset.Pparameter_maps(:,:,2)+ h;    % K-Factor
        h_parset.Pparameter_maps(:,:,3) = g*h_parset.Pparameter_maps(:,:,3);       % Shadow-Fading
        h_parset.Pparameter_maps(:,:,4) = d*h_parset.Pparameter_maps(:,:,4) + c;   % ASD
        h_parset.Pparameter_maps(:,:,5) = f*h_parset.Pparameter_maps(:,:,5) + e;   % ASA
        h_parset.Pparameter_maps(:,:,6) = m*h_parset.Pparameter_maps(:,:,6) + l;   % ESD
        h_parset.Pparameter_maps(:,:,7) = o*h_parset.Pparameter_maps(:,:,7) + n;   % ESA
        
        h_parset.Pmap_valid = true;
    else
        h_parset.Pparameter_maps = ones( no_map_y , no_map_x , 7 );
        
        a = h_parset.scenpar.DS_mu;
        b = h_parset.scenpar.AS_D_mu;
        c = h_parset.scenpar.AS_A_mu;
        d = h_parset.scenpar.KF_mu;
        e = h_parset.scenpar.ES_D_mu;
        f = h_parset.scenpar.ES_A_mu;

        % Scale maps so that their valus correspond to the "mu" values in
        % scenpar.
        h_parset.Pparameter_maps(:,:,1) = a*h_parset.Pparameter_maps(:,:,1);   % Delay Spread
        h_parset.Pparameter_maps(:,:,2) = d*h_parset.Pparameter_maps(:,:,2);   % K-Factor
        h_parset.Pparameter_maps(:,:,3) = 0*h_parset.Pparameter_maps(:,:,3);   % Shadow-Fading
        h_parset.Pparameter_maps(:,:,4) = b*h_parset.Pparameter_maps(:,:,4);   % ASD
        h_parset.Pparameter_maps(:,:,5) = c*h_parset.Pparameter_maps(:,:,5);   % ASA
        h_parset.Pparameter_maps(:,:,6) = e*h_parset.Pparameter_maps(:,:,6);   % ESD
        h_parset.Pparameter_maps(:,:,7) = f*h_parset.Pparameter_maps(:,:,7);   % ESA
        
        h_parset.Pmap_valid = false;
    end
    h_parset.Pdata_valid = false;
    
else
    if verbose && initialize
        for n = 1:vb_dots
            fprintf('o');
        end
    end
end

if nargin < 3 && verbose && initialize
    fprintf('] %5.0f seconds\n',round( etime(clock, tStart) ));
end

end

