function [ pl, scale_sf ] = get_pl( h_parset, evaltrack, i_mobile )
%GET_PL Calculates the path loss
%
%   pl = GET_PL 
%   calculates the path loss (PL) for all user positions. This function
%   implements the path loss models defined by the 'plpar' property of the
%   class 'parameter_set'.
%
%   [ pl , scale_sf ] = GET_PL 
%   In some scenarios, the SF might change with increasing distance between
%   Tx and Rx. Hence, the shadow fading provided by the parameter map has
%   to be changed accordingly. The second output parameter "scale_sf" can
%   be used for scaling the (logarithmic) SF value from the map.
%
%   [ pl , scale_sf ] = GET_PL( evaltrack ) 
%   returns the PL along the positions of the provided track 'evaltrack'
%   instead of using the positions in the class property.
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


if numel(h_parset) > 1
    error('??? Found an array of "parameter_set" objects. This is not supported for "get_pl".')
end

drifting_precision = h_parset.simpar.drifting_precision;
if nargin ~= 3
    drifting_precision = 0;
end

n_rx = 1;
n_tx = 1;
o_tx = 1;

txpos = h_parset.tx_position;

% Check if "i_mobile" is given
if exist( 'i_mobile','var' ) && ~isempty( i_mobile )
    if ~( isnumeric(i_mobile) && all( mod(i_mobile,1)==0 ) &&...
            min(i_mobile) > 0 && max(i_mobile)<=h_parset.no_positions )
        error('??? "i_mobile" is invalid.')
    end
else
    i_mobile = 1 : h_parset.no_positions;
end


if exist( 'evaltrack','var' ) && ~isempty( evaltrack )

    if ~( isa(evaltrack, 'track') )
        error('??? "evaltrack" must be of class "track".')
    end
    
    rxpos = [ evaltrack.positions(1,:) + evaltrack.initial_position(1) ; ...
        evaltrack.positions(2,:) + evaltrack.initial_position(2) ; ...
        evaltrack.positions(3,:) + evaltrack.initial_position(3) ];
    
    use_track = true;
    n_snapshots = evaltrack.no_snapshots;
    o_snapshots = ones(1,n_snapshots);
    
    if drifting_precision == 3
        if numel( i_mobile ) > 1
            i_mobile = 1;
        end
        
        e_rx = h_parset.rx_array(i_mobile).element_position;
        e_tx = h_parset.tx_array.element_position;
        
        n_rx = h_parset.rx_array(i_mobile).no_elements;
        n_tx = h_parset.tx_array.no_elements;
        
        o_rx = ones(1,n_rx);
        o_tx = ones(1,n_tx);
        
        gdir = evaltrack.ground_direction;
        c_gdir = cos(gdir);
        s_gdir = sin(gdir);
        
        e_rx_x = e_rx(1,:);
        e_rx_y = e_rx(2,:);
        e_rx_z = e_rx(3,:);
        
        % Apply the rotation
        q_s = zeros( 3,n_snapshots,n_rx );
        for i_rx = 1:n_rx
            q_s(1,:,i_rx) = c_gdir.*e_rx_x(i_rx) - s_gdir.*e_rx_y(i_rx);
            q_s(2,:,i_rx) = s_gdir.*e_rx_x(i_rx) + c_gdir.*e_rx_y(i_rx);
            q_s(3,:,i_rx) = e_rx_z(i_rx);
        end
        
        rxpos = q_s + rxpos(:,:,o_rx);
        txpos = reshape(e_tx,3,1,n_tx) + txpos(:,1,o_tx);
    end
    
else % No evaltrack given
    
    rxpos = h_parset.positions( :,i_mobile );
    use_track = false;
    n_snapshots = numel( i_mobile );
    o_snapshots = ones(1,n_snapshots);
    
end


MsBsDistance = zeros(n_snapshots, n_rx, n_tx);
for i_rx = 1:n_rx
    for i_tx = 1:n_tx
        MsBsDistance(:,i_rx,i_tx) = ...
            sqrt( sum( ( rxpos(:,:,i_rx) - txpos(:,o_snapshots,i_tx) ).^2 ) );
    end
end
MsBsDistance = reshape( MsBsDistance , 1 , [] );

% Initialize output varaibles
pl = zeros(size(MsBsDistance));
sf_sigma = pl;

% This implements the path pl models
if isfield( h_parset.plpar , 'model' )
    
    % Read commonly used data structures from class object (increases speed)
    par   = h_parset.plpar;
    CenterFrequency = h_parset.simpar.center_frequency/1e9;      % in GHz
    
    switch h_parset.plpar.model
        
        case 'logdist'
            pl          = par.A * log10(MsBsDistance) + par.B + par.C * log10(CenterFrequency);
            sf_sigma    = par.SF;
            
        case 'logdist_simple'
            pl          = par.A * log10(MsBsDistance) + par.B;
            sf_sigma    = h_parset.scenpar.SF_sigma;
            
        case 'constant'
            pl = par.A * ones( size(MsBsDistance) );
            sf_sigma    = h_parset.scenpar.SF_sigma;
            
        case 'winner_los'
            % From WINNER+ D5.3 pp. 74
            
            % Ground Distance
            dg = zeros(n_snapshots, n_rx, n_tx);
            for i_rx = 1:n_rx
                for i_tx = 1:n_tx
                    dg(:,i_rx,i_tx) = ...
                        sqrt( sum( ( rxpos(1:2,:,i_rx) - txpos(1:2,o_snapshots,i_tx) ).^2 ) );
                end
            end
            dg = reshape( dg , 1 , [] );
            
            hBS = txpos(3);
            hMS = reshape( rxpos(3,:,:,o_tx) , 1 ,[] );
            
            hBS( hBS < 1.5  ) = 1.5;
            hMS( hMS < 1.5  ) = 1.5;
            
            % Calculate the breakpoint
            G = par.B1 + par.C1*log10(CenterFrequency) + par.D1*log10(hBS) + par.E1*log10(mean(hMS));
            H = par.B2 + par.C2*log10(CenterFrequency) + par.D2*log10(hBS) + par.E2*log10(mean(hMS));
            bp  = 10^( (H-G)/( par.A1-par.A2 ) );
            
            ind = dg<=bp;
            if sum(ind)>0
                pl(ind) = par.A1*log10(dg(ind)) + par.B1 + par.C1*log10(CenterFrequency)...
                    + par.D1*log10(hBS) + par.E1*log10(hMS(ind)) + par.F1*hMS(ind);
                sf_sigma(ind) = par.sig1;
            end
            
            ind = ~ind;
            if sum(ind)>0
                pl(ind) = par.A2*log10(dg(ind)) + par.B2 + par.C2*log10(CenterFrequency)...
                    + par.D2*log10(hBS) + par.E2*log10(hMS(ind)) + par.F2*hMS(ind);
                sf_sigma(ind) = par.sig2;
            end
            
        case 'winner_nlos'
            % From WINNER+ D5.3 pp. 74
            
            % Ground Distance
            dg = zeros(n_snapshots, n_rx, n_tx);
            for i_rx = 1:n_rx
                for i_tx = 1:n_tx
                    dg(:,i_rx,i_tx) = ...
                        sqrt( sum( ( rxpos(1:2,:,i_rx) - txpos(1:2,o_snapshots,i_tx) ).^2 ) );
                end
            end
            dg = reshape( dg , 1 , [] );
            
            hBS = txpos(3);
            hMS = reshape( rxpos(3,:,:,o_tx) , 1 ,[] );
            
            hBS( hBS < 1.5  ) = 1.5;
            hMS( hMS < 1.5  ) = 1.5;
            
            if CenterFrequency < 1.5
                pl = ( par.A1 + par.Ah1 * log10( hBS ))*log10(dg) + par.B1 + ...
                    par.C1*log10(CenterFrequency) + ...
                    par.D1*log10(hBS) + ...
                    par.E1*log10(hMS) + ...
                    par.F1*hMS;
                
            elseif CenterFrequency >= 1.5 && CenterFrequency < 2
                pl = ( par.A2 + par.Ah2 * log10( hBS ))*log10(dg) + par.B2 + ...
                    par.C2*log10(CenterFrequency) + ...
                    par.D2*log10(hBS) + ...
                    par.E2*log10(hMS) + ...
                    par.F2*hMS;
                
            else % CenterFrequency >= 2
                pl = ( par.A3 + par.Ah3 * log10( hBS ))*log10(dg) + par.B3 + ...
                    par.C3*log10(CenterFrequency) + ...
                    par.D3*log10(hBS) + ...
                    par.E3*log10(hMS) + ...
                    par.F3*hMS;
            end
            sf_sigma = h_parset.scenpar.SF_sigma;
            
        case 'winner_pathloss'
            % See WINNER II D1.1.2 V1.2 (2007-09) p43 Equation (4.23)
            % PL/[dB] = A log10(d/[m]) + B + C log10(fc/[GHz]/5) + X
            
            pl = par.A * log10(MsBsDistance) + par.B + par.C * log10(h_parset.simpar.center_frequency/5e9) + par.X;
            sf_sigma = par.SF;
            
        otherwise
            sf_sigma = ones(size(sf_sigma)) * h_parset.scenpar.SF_sigma;
    end
else
    sf_sigma = ones(size(sf_sigma)) * h_parset.scenpar.SF_sigma;
end

pl = reshape(pl,n_snapshots,n_rx,n_tx);

if use_track
    sf_sigma = mean(sf_sigma);
end

% The shadow fading might change with distance. Hence, if
% the value did change, we have to rescale the values from
% the map.
SF_sigma_scenpar = h_parset.scenpar.SF_sigma;
if SF_sigma_scenpar == 0
    scale_sf = 1;
else
    scale_sf = sf_sigma / SF_sigma_scenpar;
end

end
