function deg = calc_polarization_rotation( aoa,eoa,Qr,Vr,Hr,xpr )
%CALC_POLARIZATION_ROTATION Calculates the polarization rotation
%
%   Array-dimensions:
%       aoa, eoa        1 x n_cluster x n_subpaths
%                       1 x n_cluster x n_subpaths x n_rx x n_tx
%
%       Qr              3 x n_rx 
%
%       Vr, Hr          1 x n_cluster x n_subpaths x n_rx
%                       1 x n_cluster x n_subpaths x n_rx x n_tx
% 
%       xpr             1 x 1 x n_cluster x n_subpaths
%
%       deg             n_rx , n_tx , n_cluster , n_subpaths
%
%   Reference: Jaeckel, S.; Börner, K.; Thiele, L. & Jungnickel, V. 
%   A Geometric Polarization Rotation Model for the 3D Spatial 
%   Channel Model IEEE Trans. Antennas Propag. (accepted for publication), 2012
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
% 
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published 
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

n_cluster  = size(aoa,2);
n_subpaths = size(aoa,3);
n_rx = size( Qr,2 );
o_rx = ones( 1,n_rx );

if numel( size(aoa) ) > 3
    high_res = true;
    n_tx = size(aoa,5);
    n_elements = n_tx*n_rx*n_cluster*n_subpaths;
else
    high_res = false;
    n_tx = 1;
    n_elements = n_cluster*n_subpaths;
end
o_tx = ones( 1,n_tx );

% To simplify the computations, we rotate the coordinate system
% such that the wave travel direction lies in y-direction. Thus, we need
% to rotate the orientation vector Or by -aoa(i_snapshot,i_cluster,:) - pi/2 in azimuth
% direction and eoa( i_snapshot,i_cluster,: ) in elevation direction.

pc = reshape( -aoa - pi/2 , 1,[] );
ps = sin( pc );
pc = cos( pc );

qc = reshape( eoa ,1,[] );
qs = sin( qc );
qc = cos( qc );

r = [ reshape( [ pc ; qc.*ps ; qs.*ps ] , [],1 ) , ...
    reshape( [ -ps ; qc.*pc ; qs.*pc ] , [],1 ) , ...
    reshape( [ zeros(1,n_elements) ; -qs ; qc ] , [],1 ) ];
   
    % Low Res:   3 * 20 x 3
    % High Res:  3 * 20 * n_rx * n_tx x 3

% We calculate the projection of the receiver orientation vector on
% the projection plane. Since the projection plane lies now in
% the x-z-plane due to the rotation of the coordinate system, we
% simply omit the y component of Or, switch the x and z component
% and normalize the resulting vector to unit length. The switching
% is done to obtain the same orientation as in the antenna pattern.
% "bp" is the angle between the x-axis and the projection vector.

if high_res
    tmp = reshape( r , 3*n_cluster*n_subpaths , n_rx , n_tx , 3 );
    
    Or = zeros( 3*n_cluster*n_subpaths*n_tx , n_rx );
    for n = 1:n_rx
        Or(:,n) = reshape( tmp(:,n,:,:) , [],3 ) * Qr(:,n);
    end
    
    Or = reshape( Or , 3 , n_cluster , n_subpaths , n_tx , n_rx );
    Or = permute( Or , [ 5,4,2,3,1 ] );
else
    Or = r * Qr;
    Or = reshape( Or , 3 , n_cluster , n_subpaths , n_rx );
    Or = permute( Or , [ 4,5,2,3,1 ] );
end

tmp = sqrt( Or(:,:,:,:,3).^2 + Or(:,:,:,:,1).^2 );
tmp( tmp==0 ) = 1;
Or = Or./tmp(:,:,:,:,[1 1 1]);

bp = atan2( Or(:,:,:,:,1) , Or(:,:,:,:,3) );
if ~high_res
    bp = bp( :,o_tx,:,: );
end

% We obtain the angles bt and br from the field patterns of the
% transmitter and receiver, respectively. b is the angle between the
% x-axis of the polarization plane and the polarization vector.

if high_res
    br = permute( atan2( real(Hr) , real(Vr)) , [4,5,2,3,1] );
else
    br = permute( atan2( real(Hr) , real(Vr)) , [4,1,2,3] );
    br = br(:,o_tx,:,:);
end

% The difference between the angles bt and br is the polarization
% mismatch between the receiver and the transmitter if both antenna
% elements were aligned on the same optical axis. The angle a, however
% takes the different orientation of the receive antenna into account.
% We compensate for that by introducing a virtual polarizer with rotation
% angle deg.

deg = bp - br - xpr(o_rx,o_tx,:,:);

end

