function h_channel = get_los_channels( h_parset )
%GET_LOS_CHANNELS Generates channel coefficients for the LOS path only
%
%   h_channel = GET_LOS_CHANNELS( h_parset ) generates static coefficients for the
%   LOS path only. This includes the following properties:
%
%     - antenna patterns
%     - polarization rotation for the LOS path
%     - plane-wave approximation of the phase
%     - path loss
%     - shadow fading
%
%    No further features of QuaDRiGa are used (i.e. no drifting, spherical
%    waves, time evolution, multipath fading etc.). This function can thus be
%    used to acquire quick previews of the propagation conditions for a given
%    layout.
%
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

if numel( unique(h_parset.rx_array) ) > 1
    warning('MATLAB:channel_builder',...
            ['There is more than one Rx antenna arrays in "h_parset".\n ',...
            '"get_los_channels" can only use one array. Results might be erroneous.']);
end

n_positions = h_parset.no_positions;
n_tx = h_parset.tx_array.no_elements;
n_rx = h_parset.rx_array(1).no_elements;

wave_no = 2*pi/h_parset.simpar.wavelength;

% Extract the travel directions at the initial position
gdir = zeros(1,n_positions);
if numel( h_parset.rx_track ) == n_positions
    for i_pos = 1:n_positions
        if isempty( h_parset.rx_track(i_pos).ground_direction )
            h_parset.rx_track(i_pos).compute_directions;
        end
        [~,ind] = min( sum( h_parset.rx_track(i_pos).positions.^2 ) );
        gdir(i_pos) = h_parset.rx_track(i_pos).ground_direction( ind );
    end
end

% Interpolate patterns
angles = h_parset.get_angles*pi/180;
[ Vt,Ht,CPt,Pt ] = h_parset.tx_array.interpolate( angles(1,:) , angles(3,:) );
[ Vr,Hr,CPr,Pr ] = h_parset.rx_array(1).interpolate( angles(2,:)-gdir , angles(4,:) );

% Calculate polarization rotation
aoa = reshape( angles(2,:) , 1 , 1 , n_positions );
eoa = reshape( angles(4,:) , 1 , 1 , n_positions );
Qr  = h_parset.rx_array(1).pol_vector;
pVr = reshape( Vr , 1 , 1 , n_positions , n_rx );
pHr = reshape( Hr , 1 , 1 , n_positions , n_rx );
xpr = zeros(1,1,1,n_positions);

deg = calc_polarization_rotation( aoa,eoa,Qr,pVr,pHr,xpr );
if h_parset.simpar.drifting_precision < 2
    deg = deg(:,ones(1,n_tx),:,:);
end
deg = reshape( deg , n_rx*n_tx , n_positions );

% Calculate the channel coefficients including polarization
c = zeros( n_rx*n_tx , n_positions );
for i_rx = 1 : n_rx
    for i_tx = 1 : n_tx
        ind = (i_tx-1)*n_rx + i_rx;
        
        % Antenna Patterns
        PatTx = [ reshape( Vt(1,:,i_tx) , 1,n_positions ) ;...
            reshape( Ht(1,:,i_tx) , 1,n_positions ) ];
        PatRx = [ reshape( Vr(1,:,i_rx) , 1,n_positions ) ;...
            reshape( Hr(1,:,i_rx) , 1,n_positions ) ];
        
        % LOS Polarization
        xprmat(1,:) = cos(deg(ind,:));
        xprmat(2,:) = -sin(deg(ind,:));
        xprmat(3,:) = -xprmat(2,:);
        xprmat(4,:) = xprmat(1,:);
        
        % Coefficients and antenna-dependent phase offset
        c(ind,:) = sum( [ sum( PatTx .* xprmat([1 3],:)) ;...
            sum( PatTx .* xprmat([2 4],:))] .* PatRx ) .* ...
            exp( -1j*(  wave_no*( Pt(1,:,i_tx) + Pr(1,:,i_rx) ) -...
            CPt(1,:,i_tx) - CPr(1,:,i_rx) ));
    end
end

% Apply path loss
[ path_loss , scale_sf ] = h_parset.get_pl;
rx_power = -path_loss.' + 10*log10( h_parset.sf ) .* scale_sf;
rx_power = sqrt( 10.^( 0.1 * rx_power ) );

c = c.*rx_power( ones(1,n_tx*n_rx) , : );

% Apply antenna coupling
c = reshape( c , n_rx , n_tx , n_positions );
Ct = h_parset.tx_array(1).coupling;
Cr = h_parset.rx_array(1).coupling.';

% This is only needed, if the coupling matrix is not an identity
% matrix.
if all(size(Ct) == [ n_tx , n_tx ]) && ...
        all(all( Ct - eye(n_tx) < 1e-10 )) && ...
        all(size(Cr) == [ n_rx , n_rx ]) && ...
        all(all( Cr - eye(n_rx) < 1e-10 ))
    d = c;
else
    d = zeros( size(Cr,1) , size(Ct,2) , n_positions   );
    for n = 1:n_positions
        d(:,:,n) = Cr * c(:,:,n) * Ct;
    end
end

% Create output channel object
h_channel = channel( reshape(d,size(d,1), size(d,2),1,n_positions) ,...
    zeros(n_positions,1) , 1 , false  );
h_channel.name = h_parset.name;
h_channel.tx_position = h_parset.tx_position;
h_channel.rx_position = h_parset.positions;

end