function han = visualize(h_array, element, han)
%VISUALIZE Create a plot showing the element configurations
%
%   VISUALIZE plots the 3D beam patterns of all antenna elements within the
%             antenna array.
%   VISUALIZE(element) only plots the beam patterns of the antenna elements 
%             specified by the given element indices contained in 'element'.
%
% You can change the the titles by manipulating the output handles.
%    set(han(1,10),'String','Element New')
%    set(han(1,11),'String','Vertical-New')
%    set(han(1,12),'String','Horizontal-New')
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

if nargin == 3
    update_plot = true;
else
    update_plot = false;
end

if nargin == 1
    element = 1:h_array.no_elements;
end

if ~(any(size(element) == 1) && isnumeric(element) ...
        && isreal(element) && all(mod(element, 1) == 0) && all(element > 0))
    error('??? "element" must be integer and > 0')
elseif any(element > h_array.no_elements)
    error('??? "element" exceed "no_elements"')
end

az = h_array.azimuth_grid;
if numel(az) == 361
    indt = 1:5:361;
else
    indt = 1:numel(az);
end
az = az(indt);

elev = h_array.elevation_grid';
if numel(elev) == 181
    indp = 1:5:181;
else
    indp = 1:numel(elev);
end
elev = elev(indp);

[az_grid, elev_grid] = meshgrid(az, elev);
[Xi, Yi, Zi] = sph2cart(az_grid, elev_grid, 1);
% Xi = cos(elev_grid) .* cos(az_grid);
% Yi = cos(elev_grid) .* sin(az_grid);
% Zi = sin(elev_grid);

min_value = -20; % cut off gain in dBi
scaling = 1;

for n = 1:numel(element);
    
    E_theta = h_array.field_pattern_vertical(indp, indt, element(n));
    E_phi = h_array.field_pattern_horizontal(indp, indt, element(n));
    
    
    % calculate radiation power pattern
    P = abs(E_theta).^2 + abs(E_phi).^2;
    % normalize by max value
    P_max = max(max(P));
    P = P ./ P_max;
    gain_lin = sum(sum((cos(elev_grid))))/sum(sum(P.*cos(elev_grid)));
    gain_dbi = 10*log10(gain_lin);
    han(n, 9) = gain_dbi;

    E_theta = E_theta .* sqrt(gain_lin./P_max);
    E_phi = E_phi .* sqrt(gain_lin./P_max);
    

    
    B = h_array.pol_vector(:, element(n));
    B = B * 1.2 * scaling;
    
    if ~update_plot
        
        fs = 16;
        han(n, 1) = figure('Position', [100, 100, 400, 850], 'Name', [h_array.name ' element ', num2str(element(n))]); 
        hold on; 
        axis off;
        han(n, 10) = title(['Antenna Array Element ', num2str(element(n))], 'FontName', 'Times', 'FontSize', fs+4);
        
        %% plot theta-component of the radiation power pattern
        han(n, 2) = axes('position',[0.1 0.55 0.85 0.32]);
        % partial gain in theta direction
        P_theta = 10*log10(abs(E_theta).^2);
        % cut off values smaller than -20dBi
        rho = P_theta;
        rho(rho < min_value) = min_value;
        % normalize rho s.t. min = 0 and max = scaling
        rho = (rho - min_value) ./ (max(max(rho)) - min_value) .* scaling;
        X = rho .* Xi; 
        Y = rho .* Yi; 
        Z = rho .* Zi;
        % plot partial gain in theta direction
        han(n, 3) = surf(X, Y, Z, P_theta);
        hold on
        % plot polarization vector
        han(n, 4) = plot3([0, B(1)], [0, B(2)], [0, B(3)], 'k', 'Linewidth', 3);
        hold off
        axis equal
        axis(scaling.*[-1 1 -1 1 -1 1]);
        caxis([min_value, gain_dbi]);
        set(gca, 'xtick', (-1:1).*scaling/2);
        set(gca, 'ytick', (-1:1).*scaling/2);
        set(gca, 'ztick', (-1:1).*scaling/2);
        xlabel('x')
        ylabel('y')
        zlabel('z')
        
        han(n, 11) = title('G_\theta(\theta, \phi)', 'FontName', 'Times', 'FontSize', fs);
        view(45, 33);
        set(gca, 'FontName', 'Times', 'FontSize', fs)
        
        %% plot phi-component of the radiation power pattern
        han(n, 5) = axes('position', [0.1 0.14 0.85 0.32]);
        % partial gain in phi direction
        P_phi = 10*log10(abs(E_phi).^2);
        % cut off values smaller than -20dBi
        rho = P_phi;
        rho(rho < min_value) = min_value;
        % normalize rho s.t. min = 0 and max = 20
        rho = (rho - min_value) ./ (max(max(rho)) - min_value) .* scaling;
        X = rho .* Xi; 
        Y = rho .* Yi; 
        Z = rho .* Zi;
        % plot partial gain in phi direction
        han(n, 6) = surf(X, Y, Z, P_phi);
        hold on
        % plot polarization vector
        han(n, 7) = plot3([0, B(1)], [0, B(2)], [0, B(3)], 'k', 'Linewidth', 3);
        hold off
        axis equal
        axis(scaling.*[-1 1 -1 1 -1 1]);
        caxis([min_value, gain_dbi]);
        set(gca, 'xtick', (-1:1).*scaling/2);
        set(gca, 'ytick', (-1:1).*scaling/2);
        set(gca, 'ztick', (-1:1).*scaling/2);
        xlabel('x')
        ylabel('y')
        zlabel('z')
        han(n, 12) = title('G_\phi(\theta, \phi)', 'FontName', 'Times', 'FontSize', fs);
        view(45, 33);
        set(gca, 'FontName', 'Times', 'FontSize', fs)
        
        han(n, 8) = axes('position', [0.08 0.08 0.08 0.08]);
        set(han(n, 8), 'Visible', 'Off');
        %%
        caxis([min_value, gain_dbi]);
        h = colorbar('SouthOutside', 'XTick', min_value:3:floor(gain_dbi));
        set(h, 'position', [0.08 0.07 0.80 0.03])
        set(gca, 'FontName', 'Times', 'FontSize', fs)
        zlab = get(h, 'xlabel');
        set(zlab, 'String', 'Partial Gain in dBi', 'FontName', 'Times', 'FontSize', fs);
    else
        %% plot theta-component of the radiation power pattern
        % partial gain in theta direction
        P_theta = 10*log10(abs(E_theta).^2);
        % cut off values smaller than -20dBi
        rho = P_theta;
        rho(rho < min_value) = min_value;
        % normalize rho s.t. min = 0 and max = scaling
        rho = (rho - min_value) ./ (max(max(rho)) - min_value) .* scaling;
        X = rho .* Xi; 
        Y = rho .* Yi; 
        Z = rho .* Zi;
        
        % plot partial gain in theta direction
        set(han(n,3), 'Xdata', X, 'Ydata', Y, 'Zdata', Z, 'Cdata', P_theta);
        % plot polarization vector
        set(han(n, 4), 'Xdata', [0, B(1)], 'Ydata', [0, B(2)], 'Zdata', [0, B(3)]);
        %% plot phi-component of the radiation power pattern
        % partial gain in phi direction
        P_phi = 10*log10(abs(E_phi).^2);
        % cut off values smaller than -20dBi
        rho = P_phi;
        rho(rho < min_value) = min_value;
        % normalize rho s.t. min = 0 and max = 20
        rho = (rho - min_value) ./ (max(max(rho)) - min_value) .* scaling;
        X = rho .* Xi; 
        Y = rho .* Yi; 
        Z = rho .* Zi;
        % plot partial gain in phi direction
        set(han(n, 6), 'Xdata', X, 'Ydata', Y, 'Zdata', Z, 'Cdata', P_phi);
        % plot polarization vector
        set(han(n, 7), 'Xdata', [0, B(1)], 'Ydata', [0, B(2)], 'Zdata', [0, B(3)]);

        drawnow
    end   
end

end

