function cp = rotate_pattern( h_array, deg, rotaxis, element, usage )
%ROTATE_PATTERN Rotates antenna patterns
%
%   cp = ROTATE_PATTERN matches the pattern to the orientation vector. Use the
%   function "estimate_pol_vector" to determine best matching orientation
%   vectors first. 
%
%   ROTATE_PATTERN( deg ) rotates the beam patterns of all elements around
%   the y-Axis in cartesian coordinates. The value deg is the rotation
%   angle in degrees ranging from -180 to 180°.
%
%   ROTATE_PATTERN( deg,rotaxis ) also specifies the rotation axis x,y or z.
%   ROTATE_PATTERN( deg,rotaxis,element ) rotates only the given element.
%
%   ROTATE_PATTERN( deg,rotaxis,element,usage )
%   The optional parameter 'usage' can limit the rotation procedure either
%   to the pattern or polarization. Possible values are:
%
%       0: Rotate both (pattern+polarization) - default
%       1: Rotate only pattern
%       2: Rotate only polarization
%
%   Pattern rotation provides the option to assemble antenna arrays out of single
%   elements. By setting the element_position property of an array object, elements
%   can be placed at different coordinates. In order to freely design arbitrary
%   array configurations, however, elements often need to be rotated (e.g. to
%   assemble a +/- 45° crosspolarized array out of single dipoles). This
%   functionality is provided here.
%
%   Note: Calling "rotate_pattern" will always remove the common phase
%   from the field patterns. Call "estimate_common_phase" before calling 
%   "rotate_pattern" to extract the common phase information.
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input arguments
if exist('deg','var')
    if ~( all(size(deg) == [1 1]) && isnumeric(deg) ...
            && isreal(deg) )
        error('??? "deg" must be integer and real')
    end
else
    deg = 0;
end

if exist('rotaxis','var')
    if ~( ischar(rotaxis) && ...
            (strcmp(rotaxis,'x') || strcmp(rotaxis,'y')  || strcmp(rotaxis,'z')) )
        error('??? "rotaxis" can only be x,y or z.')
    end
else
    rotaxis = 'y';
end

if exist('element','var')
    if ~( size(element,1) == 1 && isnumeric(element) ...
            &&  all( mod(element,1)==0 ) && min(element) > 0 && max(element)<=h_array.no_elements )
        error('??? "element" must be integer > 0 and can not exceed array size')
    end
else
    element = 1:h_array.no_elements;
end

if exist('usage','var')
    if ~( all(size(usage) == [1 1]) && isnumeric(usage) ...
            && any(usage == [0,1,2]) )
        error('??? "usage" must be 0,1 or 2')
    end
else
    usage = 0;
end

% Get the angles.
phi   = h_array.azimuth_grid;
theta = h_array.elevation_grid';
no_az = h_array.no_az;
no_el = h_array.no_el;

% Rotation vectors are given in degree, but calculations are done in rad.
deg = deg/180*pi;

% Rotations aroe only allows axis-wise where for each axis, another
% rotation matrix is given.
switch rotaxis
    case 'x'
        Rx = makehgtform('xrotate', deg);
        rot = Rx(1:3, 1:3);
    case 'y'
        Ry = makehgtform('yrotate', deg);
        rot = Ry(1:3, 1:3);
    case 'z'
        Rz = makehgtform('zrotate', deg);
        rot = Rz(1:3, 1:3);
end

% When the rotation angle is set to 0, we do not need to interpolate the
% pattern. However, the polarization still needs to be considered. In this
% case, "rotate_pattern" matches the pattern to the orientation vector and
% calculates the common pahse.
if deg == 0
    usage = 2;
end

% Calculate the updated angles
if usage == 0 || usage == 1
    % Calculate the transformation matrices for transforming the pattern from
    % polar coordinates to cartesian coordinates.
    % [phi1, theta1] = meshgrid(phi, theta);
    % [B(1,:,:), B(2,:,:), B(3,:,:)] = sph2cart(phi1, theta1, 1);
    
    B(1,:,:) = cos(theta)*cos(phi);             % Matrix-vector notation is faster
    B(2,:,:) = cos(theta)*sin(phi);             % ... then meshgrid and sph2cart
    B(3,:,:) = sin(theta)*ones(1,no_az);
    
    A = rot.' * reshape(B, 3, []);
    A = reshape( A.' , no_el,no_az,3 );
    
    % Fix numeric bounds
    A(A>1) = 1;
    A(A<-1) = -1;
    
    % Calculate new angles
    [phi_new, theta_new] = cart2sph( A(:,:,1), A(:,:,2), A(:,:,3) ) ;
    
    % Angles might become complex, if the values in A are out of bound due
    % to numeric offsets
    phi_new = real( phi_new );
    theta_new = real( theta_new );
    
    % We have to avoid zeros in the pattern since the interpolation becomes
    % singular at those points. We thus limit the minimum angle to a
    % certain error bound.
    err_limit = 1e-5;
    
    s = phi_new( abs(phi_new)<err_limit );
    s(s==0) = 1;
    phi_new( abs(phi_new)<err_limit ) = sign(s)*err_limit;
    
    s = theta_new( abs(theta_new)<err_limit );
    s(s==0) = 1;
    theta_new( abs(theta_new)<err_limit ) = sign(s)*err_limit;
end

if nargout == 1
   cp = zeros(no_el,no_az,numel(element));
end

for n = 1:numel(element)
    
    % When we have the angles from the projection, we interpolate the
    % 3D field patterns to get the new (rotated) patterns. This is done
    % by the interpolation routine.
    if usage == 0 || usage == 1
        [V,H,CP] = h_array.interpolate( phi_new, theta_new , element(n) );
        h_array.element_position(:,element(n)) = rot*h_array.element_position(:,element(n));
    else
        V = h_array.field_pattern_vertical(:,:,element(n));
        H = h_array.field_pattern_horizontal(:,:,element(n));
        CP = h_array.common_phase(:,:,element(n));
    end
    
    % The orientation vector
    Ov = h_array.pol_vector(:,element(n));
    Ov = rot*Ov;
    
    if usage == 0 || usage == 2
        % Here, we update the polarization of the rotated antenna
        % pattern. However, this method os only valid for linear
        % polarization.
        %
        % There can be a phase component for two reasons:
        %    1. A displacement of the antenna element, feeder cables etc.
        %    2. Polarization
        % Since we want to evaluate the polarizazion only, we calibrate the
        % displacement effect.
        
        % We determine the common phase. However, there is an ambiguity
        % concerning the sign of the polarization. This can not be solved
        % here. Hence, we only obtatin an initial common phase.
        
        indV = abs(V) > abs(H);             % The indices, where V is larger than H
        
        ang = zeros( size(V) );             % Calculate the angles
        ang(indV)  = angle(V(indV));
        ang(~indV) = angle(H(~indV));
        ang = exp( 1j * ang );              % The calibration coefficient
        
        % Determine the real parts of the patterns
        rV  = real( V./ang );               % Real part of V
        rH  = real( H./ang );               % Real part of H
        
        % If the antenna is linearly polarized, then iV abd iH are 0.
        % If there is an out of phase ecomponent (elliptical polarization),
        % then both will be > 0 at some angles.
        
        % Determine the polarization offset based on the orientation vector
        pc = -phi - pi/2;
        ps = sin( pc );
        pc = cos( pc );
        
        a  = zeros( no_el,no_az );          % The polarization rotation angle
        for o = 1:no_el
            
            qc = theta(o);
            qs = sin( qc );
            qc = cos( qc );
            
            r = [ reshape( [ pc ; qc*ps ; qs*ps ] , 3*no_az,1 ) , ...
                reshape( [ -ps ; qc*pc ; qs*pc ] ,  3*no_az,1 ) , ...
                reshape( [ zeros(1,no_az) ; -ones(1,no_az)*qs ; ones(1,no_az)*qc ] ,...
                3*no_az,1 ) ];
            
            Or = r * Ov;
            Or = reshape( Or , 3 , no_az );
            
            a(o,:)  = atan2( Or(1,:) , Or(3,:) );
        end
        
        vartheta = -atan2( rH , rV ) - a;
        qc = cos(vartheta);
        qs = sin(vartheta);
        
        patV = qc.*V - qs.*H;       % Includes the imag part and the common phase
        patH = qs.*V + qc.*H;
        
        patV = patV./ang;           % Remove the initial common phase
        patH = patH./ang;
        
        % Ideally, patV and patH should now be real-valued only.
        % The initial common phase is calibrated out of the pattern. 
        % In the channel builder, we use only the real part of the pattern
        % to determine to polarization state.
        
        if deg == 0
            % The correct common phase is obtained after splitting
            % polarization and placement effects. However, that only works
            % when the rotation angle is 0. In this case, we use
            % "rotate_pattern" to calculate the phases.
           
            ang = zeros( size(patV) );
            ang(indV)  = angle( V(indV) ) - angle( patV(indV) );
            ang(~indV) = angle( H(~indV) ) - angle( patH(~indV) );
            ang = angle( exp(1j*ang) );

            if nargout == 1
                cp(:,:,n) = ang;
            end
        end
        
        h_array.field_pattern_vertical(:,:,element(n))   = patV;
        h_array.field_pattern_horizontal(:,:,element(n)) = patH;
        h_array.common_phase(:,:,element(n)) = CP;
        h_array.pol_vector(:,element(n)) = Ov;
        
    else % mode 1 - Rotate patterns only, but not the polarization
        
        h_array.field_pattern_vertical(:,:,element(n)) = V;
        h_array.field_pattern_horizontal(:,:,element(n)) = H;
        h_array.field_pattern_horizontal(:,:,element(n)) = CP;
        h_array.pol_vector(:,element(n)) = Ov;
        
    end
end

end

