function [ h_array, mse, mse_pat ] = import_pattern( fVi, fHi , correct_phase , ...
    accuracy , max_num_elements , azimuth_grid , elevation_grid , verbose )
%IMPORT_PATTERN Converts antenna field patterns into a QuaDRiGa array object
%
%   This function converts any antenna field pattern into a QuaDRiGa
%   antenna array object. The conversion needs at most 4 elements for each
%   input field pattern. This value is doubled if the inputs are
%   complex-valued.
%
%   The input variables are:
%
%    fVi
%      The field pattern(s) for the vertical polarization given in
%      spherical coordinates. The first dimension corresponds to the
%      elevation angle (ranging from -90 to 90 degrees). The second
%      dimension is for the azimuth angle (ranging from -180 to 180
%      degrees). The third dimension belongs to the element number. The
%      default resolution is 1 degree - hence, the default size of fVi is
%      <181x361x1>. If a different resolution is given, the optional
%      variables "azimuth_grid" and "elevation_grid" must be defined.
%
%    fHi
%      The field pattern(s) for the horizontal polarization given in
%      spherical coordinates. "fHi" can be empty if no horizontal response
%      is given. If it is given, then "fHi" must have the same size as
%      "fVi".
%
%    correct_phase
%       It is possible to remove a common phase offset for both, vertical
%       and horizontal polarization. This offset can occur when an antenna
%       was characterized in an anechoic chamber where the antenna was not
%       placed exactly in the center. Hence, there is a phase difference
%       for each angle. Default setting: 0 (No correction)
%
%    accuracy
%      The required accuracy (i.e. the target MSE) of the approximation.
%      The MSE is defined as:
%
%         N = abs( fVi ).^2 + abs( fHi ).^2;
%         offset = abs( fVi - fVo ).^2 + abs( fHi - fHo ).^2;
%         MSE = mean( offset(:) ) / N;
%
%      This variable is tracked throughout the converting process. If the
%      approximation is good enough (i.e. the MSE < accuracy), then the
%      algorithm stops and returns the output pattern. Default value: 1e-6
%
%    max_num_elements
%      This value limits the number of elements per field pattern. The
%      default value is 4 which allows a perfect approximation. However,
%      more elements require more computing time in the channel model.
%
%    azimuth_grid
%      A vector specifying the azimuth sampling points of the patterns in
%      units of radians (raging from -pi to pi). This value only needs to
%      be defined if the patterns do not have the default size.
%
%    elevation_grid
%      A vector specifying the elevation sampling points of the patterns in
%      units of radians (raging from -pi/2 to pi/2). This value only needs
%      to be defined if the patterns do not have the default size.
%
%    verbose
%      The detail of the status reports: 0 - no status reports. 1 - shows a
%      progress bar (default). 2 - shows detailed progress information.
%
%   The output variables are:
%
%     h_array
%       The QuaDRiGa antenna array object generated from the field patterns.
%
%     mse
%       The MSE (as defined above) for each pattern.
%
%     mse_pat
%       The MSE (as defined above) for each pattern given for each sample
%       point of the pattern in spherical coordinates.
%
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input: fVi
if ~exist('fVi','var') || isempty(fVi)
    error('Vertical field pattern "fVi" is not defined.')
end

% Parse input: fHi
if exist('fHi','var') && ~isempty(fHi)
    if any( size(fVi) ~= size(fHi) )
        error('Horizontal field pattern "fHi" has different size than vertical pattern "fVi".')
    end
else
    fHi = zeros( size(fVi) );
end

% Parse input: verbose
if exist('correct_phase','var') && ~isempty(correct_phase)
    if ~( isnumeric(correct_phase) || ~isnan(correct_phase) || all(size(correct_phase) == [1 1]) )
        error('"correct_phase" must be numeric.')
    end
else
    correct_phase = false;
end

% Parse input: accuracy
if exist('accuracy','var') && ~isempty(accuracy)
    if ~( isnumeric(accuracy) || ~isnan(accuracy) ||...
            all(size(accuracy) == [1 1]) || ...
            accuracy>0 )
        error('"accuracy" must be a scalar number > 0')
    end
else
    accuracy = 1e-6;
end

% Parse input: max_num_elements
if exist('max_num_elements','var') && ~isempty(max_num_elements)
    if ~( isnumeric(max_num_elements) || ~isnan(max_num_elements) ||...
            all(size(max_num_elements) == [1 1]) || ...
            any( max_num_elements == 1:4 ))
        error('"max_num_elements" must be 1,2,3 or 4.')
    end
else
    max_num_elements = 4;
end

% Parse input: azimuth_grid
if exist('azimuth_grid','var') && ~isempty(azimuth_grid)
    if ~( any( size(azimuth_grid) == 1 ) && isnumeric(azimuth_grid) && isreal(azimuth_grid) &&...
            max(azimuth_grid)<=pi && min(azimuth_grid)>=-pi )
        error('??? "azimuth_grid" must be a vector containing values between -pi and pi')
    end
elseif size(fVi,2) ~= 361
    error('??? "azimuth_grid" undefiend');
else
    azimuth_grid = (-180:180)*pi/180;
end

% Parse input: elevation_grid
if exist('elevation_grid','var') && ~isempty(elevation_grid)
    if ~( any( size(elevation_grid) == 1 ) && isnumeric(elevation_grid) && isreal(elevation_grid) &&...
            max(elevation_grid)<=pi/2 && min(elevation_grid)>=-pi/2 )
        error('??? "elevation_grid" must be a vector containing values between -pi/2 and pi/2')
    end
elseif size(fVi,1) ~= 181
    error('??? "elevation_grid" undefiend');
else
    elevation_grid = (-90:90)*pi/180;
end

% Parse input: verbose
if exist('verbose','var') && ~isempty(verbose)
    if ~( isnumeric(verbose) || ~isnan(verbose) || all(size(verbose) == [1 1]) )
        error('"verbose" must be numeric.')
    end
else
    verbose = 1;
end

% Here, we split the input patterns in real and imaginary part. The
% variable "element_index" stores, which of the split patterns belong to
% which input. The first column denots the input element number and the
% second column contains the phase shift (1 or j).
element_index = [];

no_input_elements = size(fVi,3);
no_process_elements = 0;
fVp = [];
fHp = [];

% Show status information
if verbose == 1
    fprintf('Ant. Import  [');
    vb_dots = 50;
    tStart = clock;
    m0=0;
    oo = 0;
elseif verbose == 2
    disp('Splitting input patterns into real and imaginary parts');
end

% Determine the poles of the pattern. Those points are excluded from the
% accuracy estimation since they are ambgiuous.

cf_ind = true( numel( elevation_grid ) , numel( azimuth_grid ) );
ind = abs( abs( elevation_grid ) - pi/2 ) < 1e-5;
cf_ind(ind,:) = false;
cf_ind = cf_ind(:);

% Correct phase
if correct_phase
    
    % Create a new array for the current field pattern and set the
    % elevation- and azimuth grid
    ar = array('omni');
    if size(fVi,1) ~= 181
        ar.elevation_grid = elevation_grid;
    end
    if size(fVi,2) ~= 361
        ar.azimuth_grid = azimuth_grid;
    end
    
    ar.no_elements = no_input_elements;
    
    % Set the field patterns
    ar.field_pattern_vertical = fVi;
    ar.field_pattern_horizontal = fHi;
    
    common_phase = ar.estimate_common_phase;
    fVc = ar.field_pattern_vertical;
    fHc = ar.field_pattern_horizontal;    
else
    common_phase = zeros(size(fVi));
    fVc = fVi;
    fHc = fHi;
end

for n_iel = 1:no_input_elements
    
    % Calculate the norm for the real- and the imaginary part.
    Nr = abs( real( fVc(:,:,n_iel) ) ).^2 + abs( real( fHc(:,:,n_iel) ) ).^2;
    Nr = 2*mean( Nr(cf_ind) );
    
    Ni = abs( imag( fVc(:,:,n_iel) ) ).^2 + abs( imag( fHc(:,:,n_iel) ) ).^2;
    Ni = 2*mean( Ni(cf_ind) );
    
    if Nr/Ni > accuracy
        % We have a real valued component in the pattern.
        % Add component to "element_index"
        element_index = [ element_index ; n_iel 1 ];
        no_process_elements = no_process_elements + 1;
        
        % Split pattern
        fVp(:,:,no_process_elements) = real( fVc( :,:,n_iel ) );
        fHp(:,:,no_process_elements) = real( fHc( :,:,n_iel ) );
    end
    
    if Ni/Nr > accuracy
        % We have an imaginary component in the pattern.
        % Add component to "element_index"
        element_index = [ element_index ; n_iel 1j ];
        no_process_elements = no_process_elements + 1;
        
        % Split pattern
        fVp(:,:,no_process_elements) = imag( fVc( :,:,n_iel ) );
        fHp(:,:,no_process_elements) = imag( fHc( :,:,n_iel ) );
    end
end

% Update status information
if verbose == 1
    no_obj = max_num_elements * no_process_elements;
elseif verbose == 2
    disp(['DONE ... The input data has ',num2str(no_process_elements),' components.']);
end

for i_pel = 1:no_process_elements
    
    % Update status information
    if verbose == 1
        oo = oo + 1;
        m1=ceil(oo/no_obj*vb_dots); if m1>m0;
            for m2=1:m1-m0; fprintf('o'); end; m0=m1; end;
    elseif verbose == 2
        disp(' ');
        disp(['Processing pattern ',num2str(i_pel)]);
    end
    
    % Copy the element to a loca variable.
    fVo = fVp(:,:,i_pel);
    fHo = fHp(:,:,i_pel);
    
    % Calculate the norm for the current pattern.
    No = abs( fVo ).^2 + abs( fHo ).^2;
    No = 2*mean( No(cf_ind) );
    
    % Try to estimate the orientation vecor
    if verbose == 2
        disp( 'Trying to approximate pattern with 1 element' );
    end
    
    % Create a new array for the current field pattern and set the
    % elevation- and azimuth grid
    ar = array('omni');
    if size(fVi,1) ~= 181
        ar.elevation_grid = elevation_grid;
    end
    if size(fVi,2) ~= 361
        ar.azimuth_grid = azimuth_grid;
    end
    
    % Set the field patterns
    ar.field_pattern_vertical = fVo;
    ar.field_pattern_horizontal = fHo;
    
    % If there is only a vertical component, then no furrther processing is
    % needed.
    
    NVo = abs( fVo ).^2;
    NVo = mean( NVo(cf_ind) );
    NHo = abs( fHo ).^2;
    NHo = mean( NHo(cf_ind) );
    
    if NHo / NVo > accuracy
        orientation_vector = ar.estimate_pol_vector(1,0);
    end
    
    % Apply the orientation vector - this will change the pattern!
    ar.rotate_pattern(0,'x');
    
    % The resulting pattern after applying the orientation vector
    fVr = ar.field_pattern_vertical;
    fHr = ar.field_pattern_horizontal;
    
    % The difference between the original pattern and the approximation
    fV = fVo - fVr;
    fH = fHo - fHr;
    
    % Calculate the offset and the error
    offset = abs( fV ).^2 + abs( fH ).^2;
    err = mean( offset(cf_ind) ) / No;
    
    % Update status information
    if verbose == 1 && max_num_elements >= 2
        oo = oo + 1;
        m1=ceil(oo/no_obj*vb_dots); if m1>m0;
            for m2=1:m1-m0; fprintf('o'); end; m0=m1; end;
    elseif verbose == 2
        disp( ['Accuracy: ',num2str(err)] );
    end
    
    % Try to approximate the pattern with 2 elements
    if err > accuracy && max_num_elements >= 2
        if verbose == 2
            disp( 'Trying to approximate pattern with 2 elements' );
        end
        
        ar.no_elements = 2;
        ar.field_pattern_vertical(:,:,2) = fV;
        ar.field_pattern_horizontal(:,:,2) = fH;
        ar.estimate_pol_vector(2,0);
        ar.rotate_pattern(0,'x',2);
        
        fVr = sum( ar.field_pattern_vertical,3);
        fHr = sum( ar.field_pattern_horizontal,3);
        
        fV = fVo - fVr;
        fH = fHo - fHr;
        
        offset = abs( fV ).^2 + abs( fH ).^2;
        err = mean( offset(cf_ind) ) / No;
        
        if verbose == 2
            disp( ['Accuracy: ',num2str(err)] );
        end
    end
    
    % If the error is still too high, we try to increase the number of
    % elements. With 4 elements, an error free approximation can be
    % achieved. The approximated pattern needs to fullfill 3 criteria:
    %
    %   1. It must be continuous in order to allow interpolation in the
    %      channel model later on.
    %   2. The element gain must not increase indefinitely.
    %   3. The number of elements used for the approximation has to be
    %      as small as possible within the given accuracy to reduce the
    %      simulation time.
    
    for i_step = 3:max_num_elements
        
        % Update status information
        if verbose == 1
            oo = oo + 1;
            m1=ceil(oo/no_obj*vb_dots); if m1>m0;
                for m2=1:m1-m0; fprintf('o'); end; m0=m1; end;
        end
        
        if err > accuracy
            
            if verbose == 2
                disp( ['Trying to approximate pattern with ',num2str(i_step),' elements'] );
            end
            
            % We use a filter function in order to make the output pattern
            % continuous. Here we calculate the filter coefficients. Those
            % coefficients depend on the orientation of the antenna.
            
            af = array('dipole');
            tmp = sin( (1-1e-6)*af.elevation_grid + pi/2 );
            P = zeros( af.no_el,af.no_az );
            for n = 1:af.no_az
                P(:,n) = tmp;
            end
            af.field_pattern_vertical = P;
            
            % Here we apply the reduced grid.
            
            if size(fVi,1) ~= 181 || size(fVi,2) ~= 361
                af.set_grid( azimuth_grid , elevation_grid );
            end
            
            % We change the orientation of the above pattern in order to
            % get matched filters for each antenna element.
            
            if i_step == 3
                af.copy_element(1,2:3);
                af.rotate_pattern( 120,'x',2);
                af.rotate_pattern( -120,'x',3 );
                
                if max_num_elements == 3
                    max_num_iterations = 32;
                else
                    max_num_iterations = 12;
                end
            else
                af.copy_element(1,2);
                af.rotate_pattern( 109.4714,'y',2 );
                af.copy_element(2,3);
                af.rotate_pattern( 120,'z',3);
                af.copy_element(2,4);
                af.rotate_pattern( -120,'z',4 );
                
                max_num_iterations = 32;
            end
            
            % Next, we apply the orientation vector.
            
            [az,el] = cart2sph( orientation_vector(1), orientation_vector(2),...
                orientation_vector(3) );
            az = az * 180/pi;
            el = el * 180/pi;
            
            af.rotate_pattern( 90-el,'y' );
            af.rotate_pattern( az,'z' );
            
            % Finally, we extract the filter coefficients.
            
            a1V = reshape( real( af.field_pattern_vertical(:,:,1) ) , [] , 1);
            a1H = reshape( real( af.field_pattern_horizontal(:,:,1) ) , [] , 1);
            
            a2V = reshape( real( af.field_pattern_vertical(:,:,2) ) , [] , 1);
            a2H = reshape( real( af.field_pattern_horizontal(:,:,2) ) , [] , 1);
            
            a3V = reshape( real( af.field_pattern_vertical(:,:,3) ) , [] , 1);
            a3H = reshape( real( af.field_pattern_horizontal(:,:,3) ) , [] , 1);
            
            if i_step == 4
                a4V = reshape( real( af.field_pattern_vertical(:,:,4) ) , [] , 1);
                a4H = reshape( real( af.field_pattern_horizontal(:,:,4) ) , [] , 1);
            end
            
            % Initialize the output array.
            
            ar = af.copy_objects;
            z = zeros( ar.no_el , ar.no_az  );
            for n_element = 1:ar.no_elements
                ar.field_pattern_vertical(:,:,n_element) = z;
                ar.field_pattern_horizontal(:,:,n_element) = z;
            end
            z = z(:);
            
            fV = fVo;
            fH = fHo;
            
            % Now we iteratively update the patterns of the output array
            % until the error converges to the given accuracy.
            
            err = Inf;
            iteration_cnt = 1;
            while err > accuracy && iteration_cnt <= max_num_iterations
                
                % Apply the filters for vertical polarization
                w = real( fV(:) );
                
                f1V = z;
                f1V( w>0 & a1V>0 ) = w( w>0 & a1V>0 );
                f1V( w<0 & a1V<0 ) = -w( w<0 & a1V<0 );
                f1V = f1V .* a1V;
                
                w = w-f1V;
                f2V = z;
                f2V( w>0 & a2V>0 ) = w( w>0 & a2V>0 );
                f2V( w<0 & a2V<0 ) = -w( w<0 & a2V<0 );
                f2V = f2V .* a2V;
                
                w = w-f2V;
                f3V = z;
                f3V( w>0 & a3V>0 ) = w( w>0 & a3V>0 );
                f3V( w<0 & a3V<0 ) = -w( w<0 & a3V<0 );
                f3V = f3V .* a3V;
                
                if i_step == 4
                    w = w-f3V;
                    f4V = z;
                    f4V( w>0 & a4V>0 ) = w( w>0 & a4V>0 );
                    f4V( w<0 & a4V<0 ) = -w( w<0 & a4V<0 );
                    f4V = f4V .* a4V;
                end
                
                % Apply the filters for horizontal polarization
                w = real( fH(:) );
                
                f1H = z;
                f1H( w>0 & a1H>0 ) = w( w>0 & a1H>0 );
                f1H( w<0 & a1H<0 ) = -w( w<0 & a1H<0 );
                f1H = f1H .* a1H;
                
                w = w-f1H;
                f2H = z;
                f2H( w>0 & a2H>0 ) = w( w>0 & a2H>0 );
                f2H( w<0 & a2H<0 ) = -w( w<0 & a2H<0 );
                f2H = f2H .* a2H;
                
                w = w-f2H;
                f3H = z;
                f3H( w>0 & a3H>0 ) = w( w>0 & a3H>0 );
                f3H( w<0 & a3H<0 ) = -w( w<0 & a3H<0 );
                f3H = f3H .* a3H;
                
                if i_step == 4
                    w = w-f3H;
                    f4H = z;
                    f4H( w>0 & a4H>0 ) = w( w>0 & a4H>0 );
                    f4H( w<0 & a4H<0 ) = -w( w<0 & a4H<0 );
                    f4H = f4H .* a4H;
                end
                
                % Update the patterns
                ar.field_pattern_vertical(:,:,1)   = ar.field_pattern_vertical(:,:,1) + reshape(f1V , ar.no_el , ar.no_az );
                ar.field_pattern_horizontal(:,:,1) = ar.field_pattern_horizontal(:,:,1) + reshape(f1H , ar.no_el , ar.no_az );
                
                ar.field_pattern_vertical(:,:,2)   = ar.field_pattern_vertical(:,:,2) + reshape(f2V , ar.no_el , ar.no_az );
                ar.field_pattern_horizontal(:,:,2) = ar.field_pattern_horizontal(:,:,2) + reshape(f2H , ar.no_el , ar.no_az );
                
                ar.field_pattern_vertical(:,:,3)   = ar.field_pattern_vertical(:,:,3) + reshape(f3V , ar.no_el , ar.no_az );
                ar.field_pattern_horizontal(:,:,3) = ar.field_pattern_horizontal(:,:,3) + reshape(f3H , ar.no_el , ar.no_az );
                
                if i_step == 4
                    ar.field_pattern_vertical(:,:,4)   = ar.field_pattern_vertical(:,:,4) + reshape(f4V , ar.no_el , ar.no_az );
                    ar.field_pattern_horizontal(:,:,4) = ar.field_pattern_horizontal(:,:,4) + reshape(f4H , ar.no_el , ar.no_az );
                end
                
                % Apply the orientation vectors - This will change the
                % patterns.
                ar.rotate_pattern(0,'x');
                
                % Calculate the offset and get the input for the next
                % iteration.
                fVr = sum( ar.field_pattern_vertical , 3 );
                fHr = sum( ar.field_pattern_horizontal , 3 );
                
                fV = fVo - fVr;
                fH = fHo - fHr;
                
                offset = abs( fV ).^2 + abs( fH ).^2;
                err = mean( offset(cf_ind) ) / No;
                
                if verbose == 2
                    disp( ['Iteration: ',num2str(iteration_cnt) , ' - Accuracy: ', num2str(err)] );
                end
                iteration_cnt = iteration_cnt + 1;
            end
        end
    end
    
    % Assembling output array.
    if i_pel == 1
        
        h_array = ar.copy_objects;
        h_array.field_pattern_vertical = real( ar.field_pattern_vertical );
        h_array.field_pattern_horizontal = real( ar.field_pattern_horizontal );
        h_array.coupling = ones( ar.no_elements , 1 ) * element_index( i_pel , 2 );
        
        out_element_index = ones( ar.no_elements , 1 );
        
    else
        
        start_element = h_array.no_elements + 1;
        coupling = h_array.coupling;
        
        h_array.no_elements = h_array.no_elements + ar.no_elements;
        h_array.field_pattern_vertical( :,:,start_element:end ) =...
            real( ar.field_pattern_vertical );
        h_array.field_pattern_horizontal( :,:,start_element:end ) =...
            real( ar.field_pattern_horizontal );
        h_array.pol_vector(:,start_element:end) = ar.pol_vector;
        
        coupling( start_element : start_element + ar.no_elements - 1 ,...
            element_index(i_pel,1) ) = element_index(i_pel,2);
        h_array.coupling = coupling;
        
        out_element_index = [ out_element_index;...
            ones( ar.no_elements,1 )*element_index(i_pel,1) ];
        
    end
    
end

if verbose == 2
    disp(' ');
    disp('Creating output data structure');
end

% Format Output
h_array.name = 'imported';

mse_pat = zeros( h_array.no_el , h_array.no_az , no_input_elements );
mse = zeros( no_input_elements , 1 );

% Set common phase and calculate total error
for i_iel = 1:no_input_elements
    
    % The output array elements belonging to
    ind = find( out_element_index == i_iel );
    
    % Set common phase
    if correct_phase
        h_array.common_phase(:,:,ind) = common_phase( :,:, ones(1,numel(ind))*i_iel );
    end
    
    fVr = zeros( h_array.no_el , h_array.no_az );
    fHr = zeros( h_array.no_el , h_array.no_az );
    for i_oel = 1:numel(ind)
        fVr = fVr + h_array.field_pattern_vertical( :,:,ind(i_oel) ) .*...
            h_array.coupling( ind(i_oel),i_iel );
        
        fHr = fHr + h_array.field_pattern_horizontal( :,:,ind(i_oel) ) .*...
            h_array.coupling( ind(i_oel),i_iel );
    end
    
    cp = exp( 1j*common_phase(:,:,i_iel) );
    fV = fVi(:,:,i_iel) - fVr.*cp;
    fH = fHi(:,:,i_iel) - fHr.*cp;
    
    % Calculate the norm for the current pattern.
    No = abs( fVi(:,:,i_iel) ).^2 + abs( fHi(:,:,i_iel) ).^2;
    No = 2*mean( No(cf_ind) );
    
    mse_pat(:,:,i_iel) = abs( fV ).^2 + abs( fH ).^2;
    tmp = mse_pat(:,:,i_iel);
    mse(i_iel) = mean( tmp(cf_ind) ) / No;
end

% Update progress bar and show time
if verbose == 1
    fprintf('] %5.0f seconds\n',round( etime(clock, tStart) ));
end

end

