function generate(h_array, array_type, element, phi_3dB, theta_3dB, rear_gain )
%GENERATE Generates predefined arrays
%
%   GENERATE( array_type ) generates a new, predefined antenna array given by
%   array_type. Currently supported are:
%
%   'omni'				An isotropic radiator with vertical polarization.
%
%   'dipole'/'short-dipole'		A short dipole radiating with vertical polarization.
%
%   'half-wave-dipole'	A half-wave dipole radiating with vertical polarization.
%
%   'patch'  			An ideal patch antenna with 90 deg opening in azimuth and
%           			elevation (vertical polarization).
%
%   'custom'			An antenna with a custom gain in elevation and azimuth.
%           			E.g.: a.generate('custom',1,90,90,0.1);
%           			create an array with 90 degree opening in azimuth and elevation
%           			and 0.1 rear gain. Half-Power Beamwidth in
%           			azimuth direction (phi_3dB) and in elevation
%           			direction (theta_3dB) should be given in degrees.
%           			The Rear Gain (rear_gain) should be given in
%           			linear scale.
%
%   'xpol'   			Two elements with ideal isotropic patterns (vertical
%           			polarization). The second element is tilted by 90 degree.
%
%   'rhcp-dipole'   	Two crossed dipoles with one port. The signal on the 
%           			second element (horizontal) is shifted by -90 deg out of phase.
%           			The two elements thus create a RHCP signal. 
%    
%   'lhcp-dipole'   	Two crossed dipoles with one port. The signal on the 
%           			second element (horizontal) is shifted by +90 deg out of phase.
%           			The two elements thus create a LHCP singnal. 
%
%   'lhcp-rhcp-dipole'  Two crossed dipoles. For input port 1, the signal on
%           			the second element (horizontal) is shifted by +90 deg out of
%           			phase. For input port 2, the the signal on the second element
%           			(horizontal) is shifted by -90 deg out of phase. Port 1 thus
%           			transmits a LHCP signal and port 2 trnasmits a RHCP singal.
%
%   'ulaX'    			Unified linear arrays composed of omni-antennas (vertical
%           			polarization)  with 10 cm element distance. X can be 2,4 or 8.
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

supported_types = array.supported_types;
if ~exist( 'array_type' , 'var' ) || isempty(array_type)
    array_type = 'omni';
elseif ~( ischar(array_type) && any( strcmpi(array_type,supported_types)) )
    str = ['Array type "',array_type,'" not found. Supported types are: '];
    no = numel(supported_types);
    for n = 1:no
        str = [str,supported_types{n}];
        if n<no
            str = [str,', '];
        end
    end
    error(str);
end

if ~exist( 'element' , 'var' ) || isempty(element)
    element = 0;
else
    if ~( size(element,1) == 1 && isnumeric(element) && all(size(element) == [1 1]) ...
            &&  all( mod(element,1)==0 ) && min(element) >= 0 )
        error('??? "element" must be scalar, integer >= 0 and can not exceed array size')
    end
end

if numel(h_array) > 1
    % Do for parameter_set_arrays
    for n=1:numel(h_array)
        h_array(n).generate( array_type,1 );
    end
    
else
    if element > h_array.no_elements
        h_array.no_elements = element;
    end
    
    switch array_type
        
        case 'omni'
            if element == 0 || isempty(h_array.no_elements)
                h_array.no_elements                 = 1;
                h_array.elevation_grid              = (-90:90)*pi/180;
                h_array.azimuth_grid                = (-180:180)*pi/180;
                h_array.element_position            = zeros( 3,1 );
                h_array.field_pattern_vertical      = ones( 181,361 );
                h_array.field_pattern_horizontal    = zeros( 181,361 );
                h_array.pol_vector                  = [0;0;1];
                h_array.coupling                    = 1;
            else
                h_array.element_position(:,element)           = zeros( 3,1 );
                h_array.field_pattern_vertical(:,:,element)   = ones( h_array.no_el,h_array.no_az );
                h_array.field_pattern_horizontal(:,:,element) = zeros( h_array.no_el,h_array.no_az );
                h_array.pol_vector(:,element)                 = [0;0;1];
            end
            
        case {'short-dipole', 'dipole'}
            if element == 0 || isempty(h_array.no_elements)
                h_array.generate('omni');
                h_array.interpolation_method        = 'linear';
                e = 1;
            else
                e = element;
            end
            [~, theta_grid] = meshgrid(h_array.azimuth_grid, h_array.elevation_grid);
            
            % Short dipole
            E_theta = cos( (1 - 1e-6) * theta_grid );
            E_phi = zeros(size(E_theta));
            
            P = E_theta.^2 + E_phi.^2;      % Calculate radiation power pattern
            P_max = max(max(P));            % Normalize by max value
            P = P ./ P_max;
            
            % Calculate the Gain
            gain_lin = sum(sum( cos(theta_grid) )) / sum(sum( P.*cos(theta_grid) ));
           
            % Normalize by Gain
            E_theta = E_theta .* sqrt(gain_lin./P_max);
            E_phi = E_phi .* sqrt(gain_lin./P_max);
            
            h_array.field_pattern_vertical(:,:,e) = E_theta;
            h_array.field_pattern_horizontal(:,:,e) = E_phi;
            h_array.pol_vector(:,e) = [0;0;1];
        
        case 'half-wave-dipole'
            if element == 0 || isempty(h_array.no_elements)
                h_array.generate('omni');
                h_array.interpolation_method        = 'linear';
                e = 1;
            else
                e = element;
            end
            [~, theta_grid] = meshgrid(h_array.azimuth_grid, h_array.elevation_grid);
            
            % Half-Wave dipole
            E_theta = cos( pi/2*sin((1 - 1e-6) * theta_grid )) ./ cos((1 - 1e-6) * theta_grid);
            E_theta( isnan(E_theta) ) = 0;
            E_phi = zeros(size(E_theta));
            
            P = E_theta.^2 + E_phi.^2;      % Calculate radiation power pattern
            P_max = max(max(P));            % Normalize by max value
            P = P ./ P_max;
            
            % Calculate the Gain
            gain_lin = sum(sum( cos(theta_grid) )) / sum(sum( P.*cos(theta_grid) ));
           
            % Normalize by Gain
            E_theta = E_theta .* sqrt(gain_lin./P_max);
            E_phi = E_phi .* sqrt(gain_lin./P_max);
            
            h_array.field_pattern_vertical(:,:,e) = E_theta;
            h_array.field_pattern_horizontal(:,:,e) = E_phi;
            h_array.pol_vector(:,e) = [0;0;1];
            
        case 'patch'
            if element == 0 || isempty(h_array.no_elements)
                h_array.generate('custom',0,90,90,0);
            else
                h_array.generate('custom',element,90,90,0);
            end
            
        case 'custom'
            if nargin < 4
                phi_3dB = 120;
            end
            
            if nargin < 5
                theta_3dB = 120;
            end
            
            if nargin < 6
                rear_gain = 0.1;
            end
            
            if ~( size(phi_3dB,1) == 1 && isnumeric(phi_3dB) && isreal(phi_3dB) &&...
                    all(size(phi_3dB) == [1 1]) )
                error('Azimuth HPBW (phi_3dB) has invalid value.')
            end
            
            if ~( size(theta_3dB,1) == 1 && isnumeric(theta_3dB) && isreal(theta_3dB) &&...
                    all(size(theta_3dB) == [1 1]) )
                error('Elevation HPBW (theta_3dB) has invalid value.')
            end
            
            if ~( size(rear_gain,1) == 1 && isnumeric(theta_3dB) && isreal(rear_gain) && ...
                    all(size(rear_gain) == [1 1]) && rear_gain>=0 && rear_gain<0.5)
                error('Front-to-back ratio (rear_gain) has invalid value.')
            end
            
            if element == 0 || isempty(h_array.no_elements)
                h_array.generate('omni');
                h_array.interpolation_method        = 'linear';
                e = 1;
            else
                e = element;
            end
            
            % Calculate the azimuth response
            phi = h_array.azimuth_grid;
            ind = find(phi/pi*180 >= phi_3dB/2, 1);
            
            a   = 1;        % Initial angle
            dm  = 0.5;      % Step size
            x   = inf;
            delta = Inf;
            ddir = +1;
            lp = 1;
            while lp < 5000 && delta > 1e-7
                if lp > 1
                    an = a + ddir * dm;
                    delta = abs(a-an);
                else
                    an = a;
                end
                
                C = rear_gain + (1 - rear_gain) * exp(-an * phi.^2);
                xn = abs(C(ind) - 0.5);
                
                if xn < x
                    a = an;
                    x = xn;
                else
                    ddir = -ddir;
                    dm = 0.382 * dm;
                end
                lp = lp + 1;
            end
            C = rear_gain + (1 - rear_gain) * exp(-an * phi.^2);
            
            
            % Calculate the elevation response
            theta = h_array.elevation_grid;
            ind = find(theta/pi*180 >= theta_3dB/2, 1);
            
            a   = 1;        % Initial angle
            dm  = 0.5;      % Step size
            x   = inf;
            delta = Inf;
            ddir = +1;
            lp = 1;
            while lp < 5000 && delta > 1e-7
                if lp > 1; 
                    an = a + ddir * dm; 
                    delta = abs(a-an); 
                else
                    an = a;
                end
                
                D = cos(theta).^an;
                xn = abs(D(ind) - 0.5);
                
                if xn < x
                    a = an;
                    x = xn;
                else
                    ddir = -ddir;
                    dm = 0.382 * dm;
                end
                lp = lp + 1;
            end
            D = cos(theta).^an;
            
            P = zeros(181,361);
            for a = 1:181
                for b = 1:361
                    P(a, b) = D(a) * C(b);
                end
            end
            P(P < rear_gain) = rear_gain;
            
            E_theta =  sqrt(P);
            E_phi = zeros(size(E_theta));
            
            [~, theta_grid] = meshgrid(h_array.azimuth_grid, h_array.elevation_grid);
            
            P = E_theta.^2 + E_phi.^2;      % Calculate radiation power pattern
            P_max = max(max(P));            % Normalize by max value
            P = P ./ P_max;
            
            % Calculate the Gain
            gain_lin = sum(sum( cos(theta_grid) )) / sum(sum( P.*cos(theta_grid) ));
           
            E_theta = E_theta .* sqrt(gain_lin./P_max);
            E_phi = E_phi .* sqrt(gain_lin./P_max);
            
            h_array.field_pattern_vertical(:,:,e) = E_theta;
            h_array.field_pattern_horizontal(:,:,e) = E_phi;
            
            h_array.field_pattern_vertical(:,:,e) = E_theta;
            h_array.field_pattern_horizontal(:,:,e) = zeros(h_array.no_el, h_array.no_az);
            h_array.pol_vector(:,e)       = [0;0;1];
            
        case 'xpol'
            if element ~= 0
                error('Choosing one element is not allowed when generating antennas with more than one element.')
            end
            h_array.generate('omni');
            h_array.copy_element(1,2);
            h_array.rotate_pattern(90,'x',2);
            
        case 'rhcp-dipole'
            if element ~= 0
                error('Choosing one element is not allowed when generating antennas with more than one element.')
            end
            h_array.generate('dipole');
            h_array.generate('dipole',2);
            h_array.rotate_pattern(90,'x',2);
            h_array.coupling = 1/sqrt(2) * [1;-1j];
            
        case 'lhcp-dipole'
            h_array.generate('rhcp-dipole',element);
            h_array.coupling = 1/sqrt(2) * [1;1j];
            
        case 'lhcp-rhcp-dipole'
            h_array.generate('rhcp-dipole',element);
            h_array.coupling = 1/sqrt(2) * [1 1;1j -1j];
            
        case 'ula2'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            h_array.generate('omni');
            h_array.no_elements                 = 2;
            h_array.element_position(2,:)       = [-0.05 0.05];
            
        case 'ula4'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            h_array.generate('omni');
            h_array.no_elements                 = 4;
            h_array.element_position(2,:)       = -0.15 :0.1: 0.15;
            
        case 'ula8'
            if element ~= 0
                error('Choosing one element is not allowed when generating arrays.')
            end
            h_array.generate('omni');
            h_array.no_elements                 = 8;
            h_array.element_position(2,:)       = -0.35 :0.1: 0.35;
            
    end
    
    h_array.name = array_type;
end
end

