function common_phase = estimate_common_phase( h_array , element )
%ESTIMATE_COMMON_PHASE Estimetes the common phase from the field patterns
%
%   It is possible that antenna patterns have a phase component. For
%   example, an antenna array might be assembled out of several elements,
%   which are at different position in the array. When the patterns of the
%   elements are then calibrated in the lab, the individual positions
%   result in a phase offset which is part of the measured pattern
%   response. The core function of QuaDRiGa, however, only uses the real
%   part if the provided patterns. Hence, not calibrating the phase offset
%   out of the pattern will lead to errors.       
% 
%   common_phase = ESTIMATE_COMMON_PHASE( element )
%   This function calculates the phase from the pattern and stores it in
%   the "common_phase" property of the antenna array object. However, this
%   requires that the orientation vectors are set correctly, i.e. you need
%   to call "estimate_pol_vector" first when importing measured patterns.   
%
%   With the parameter "element", it is possible to select a specific
%   element of an array for which the evaluation should be done.
%
%
% QuaDRiGa Copyright (C) 2011-2012 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


% Parse input variables.
if exist('element','var')
    if ~( size(element,1) == 1 && isnumeric(element) ...
            &&  all( mod(element,1)==0 ) && min(element) > 0 && max(element) ...
            <= h_array.no_elements )
        error('??? "element" must be integer > 0 and can not exceed array size')
    end
else
    element = 1:h_array.no_elements ;
end

% This calculates the common phase and removes the phase information from
% the field patterns.

ap = h_array.copy_objects;
cp = ap.rotate_pattern( 0 , 'x' , element , 2 );

common_phase = angle( exp( 1j*(h_array.common_phase + cp)) );
cp = exp( 1j*cp );

h_array.field_pattern_vertical(:,:,element) =...
    h_array.field_pattern_vertical(:,:,element)./cp;
h_array.field_pattern_horizontal(:,:,element) =...
    h_array.field_pattern_horizontal(:,:,element)./cp;

h_array.common_phase(:,:,element) = common_phase;

end


