function h_parset = generate_correlation_maps( h_parset, initialize, vb_dots )
%GENERATE_CORRELATION_MAPS Generates a new set of correlation maps
%
%   GENERATE_CORRELATION_MAPS manually generates the correlation maps.
%   Correlation maps are needed to handle the distance-dependant correlation
%   between mobile terminals. E.g. when two terminals are close to each other,
%   they will see similar channels.
%
% QuaDRiGa Copyright (C) 2011-2013 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

% Parse input parameters
if exist( 'initialize' , 'var' ) && ~isempty( initialize )
    if ~( all(size(initialize) == [1 1]) ...
            && (isnumeric(initialize) || islogical(initialize)) ...
            && any( initialize == [0 1] ) )
        error('??? "initialize" must be 0 or 1')
    end
else
    initialize = true;
end

verbose = h_parset(1).simpar.show_progress_bars;
if nargin < 3
    if verbose
        vb_dots = ones(1,numel(h_parset))*floor( 50/numel(h_parset) );
        tmpA = numel( vb_dots );
        tmpB = 50-sum( vb_dots );
        tmpC = 1 : floor( tmpA/tmpB ) : tmpA;
        for n = 1:tmpB
            vb_dots( tmpC(n) ) = vb_dots( tmpC(n) )+1;
        end
    else
        vb_dots = zeros(1,numel(h_parset));
    end
end

if nargin < 3 && verbose && initialize
    fprintf('Parameters   [');
    tStart = clock;
end

if numel(h_parset) > 1
    for n=1:numel(h_parset)
        generate_correlation_maps( h_parset(n), initialize, vb_dots(n) );
    end
    
elseif h_parset.no_positions > 0
    
    if ~h_parset.LSP_matrix_isOK
        error('LSP_matrix is not positive-definite.');
    end
    
    % Get the autocorrelation distances and scale them to the map resolution.
    delta = [ h_parset.scenpar.DS_lambda,...
        h_parset.scenpar.KF_lambda,...
        h_parset.scenpar.SF_lambda,...
        h_parset.scenpar.AS_D_lambda,...
        h_parset.scenpar.AS_A_lambda,...
        h_parset.scenpar.ES_D_lambda,...
        h_parset.scenpar.ES_A_lambda] .* h_parset.samples_per_meter;
    
    % The variance in the map. It it is 0, we don't need to create the map.
    sigma = [ h_parset.scenpar.DS_sigma,...
        h_parset.scenpar.KF_sigma,...
        h_parset.scenpar.SF_sigma,...
        h_parset.scenpar.AS_D_sigma,...
        h_parset.scenpar.AS_A_sigma,...
        h_parset.scenpar.ES_D_sigma,...
        h_parset.scenpar.ES_A_sigma] ~= 0;
    
    % Extract the minimum and maximum positions from the list of positions
    min_pos = floor( min( h_parset.positions-h_parset.map_extension,[],2 )...
        * h_parset.samples_per_meter)/h_parset.samples_per_meter;
    max_pos = ceil ( max( h_parset.positions+h_parset.map_extension,[],2 )...
        * h_parset.samples_per_meter)/h_parset.samples_per_meter;
    
    % Determine the map edges taking an existing grid into account
    x_min = min( [ h_parset.map_extent(1, 1) , min_pos(1) ] );
    x_max = max( [ h_parset.map_extent(1, 2) , max_pos(1) ] );
    y_min = min( [ h_parset.map_extent(2, 1) , min_pos(2) ] );
    y_max = max( [ h_parset.map_extent(2, 2) , max_pos(2) ] );
    
    % Extending the map while keeping the sampling rate constant
    h_parset.map_extent = [x_min, x_max; y_min, y_max];
    
    % Get map dimensions
    no_map_y = h_parset.map_size(2);
    no_map_x = h_parset.map_size(1);
    
    if initialize
        
        switch h_parset.simpar.use_map_algorithm
            case 1
                h_parset = generate_map_winner( h_parset, delta, sigma, vb_dots );
            case 2
                h_parset = generate_map_filter( h_parset, delta, sigma, vb_dots );
        end
        
        % Copy map data
        names = {'ds_map','kf_map','sf_map','asD_map','asA_map','esD_map','esA_map'};
        ind = find( sigma );
        maps = zeros( h_parset.map_size(2)*h_parset.map_size(1), sum(sigma) , 'single' );
        for n = 1 : sum(sigma)
            maps(:,n) = h_parset.( names{ind(n)} )(:);
        end
        
        % If there is no variance, no inter-parameter correlation needs to
        % be applied. 
        R_sqrt = sqrtm( h_parset.LSP_xcorr_matrix );
        R_sqrt = R_sqrt( sigma,sigma );
        
        % Apply inter-parameter correlation
        maps = maps * R_sqrt;
        
        % Reapply the variances after appying the correlation values
        mu  = mean( maps );
        sig = std( maps );
        for n = 1 : sum(sigma)
            maps(:,n) = (maps(:,n)-mu(n))./sig(n);
        end

        % Transform Normal distributed maps to scenario specific distributions
        % Copy correlated data back to the initial function
        cnt = 1;
        for n = 1:7
            switch n
                case 1
                    a = h_parset.scenpar.DS_mu;
                    b = h_parset.scenpar.DS_sigma;
                case 2
                    a = h_parset.scenpar.KF_mu;
                    b = h_parset.scenpar.KF_sigma;
                case 3
                    a = 0;
                    b = h_parset.scenpar.SF_sigma;
                case 4
                    a = h_parset.scenpar.AS_D_mu;
                    b = h_parset.scenpar.AS_D_sigma;
                case 5
                    a = h_parset.scenpar.AS_A_mu;
                    b = h_parset.scenpar.AS_A_sigma;
                case 6
                    a = h_parset.scenpar.ES_D_mu;
                    b = h_parset.scenpar.ES_D_sigma;
                case 7
                    a = h_parset.scenpar.ES_A_mu;
                    b = h_parset.scenpar.ES_A_sigma;
            end
            if sigma(n)
                h_parset.(names{n}) =...
                    reshape( b*maps(:,cnt) + a, no_map_y, no_map_x );
                cnt = cnt + 1;
            else
                h_parset.(names{n}) = a;
            end
        end
        h_parset.Pmap_valid = true;
        
    else
        % Initialize the maps with default parameters
        h_parset.Pmap_ds  = h_parset.scenpar.DS_mu;
        h_parset.Pmap_kf  = h_parset.scenpar.KF_mu;
        h_parset.Pmap_sf  = 0;
        h_parset.Pmap_asD = h_parset.scenpar.AS_D_mu;
        h_parset.Pmap_asA = h_parset.scenpar.AS_A_mu;
        h_parset.Pmap_esD = h_parset.scenpar.ES_D_mu;
        h_parset.Pmap_esA = h_parset.scenpar.ES_A_mu;
        
        h_parset.Pmap_valid = false;
    end
    h_parset.Pdata_valid = false;
    
else
    if verbose && initialize
        for n = 1:vb_dots
            fprintf('o');
        end
    end
end

if nargin < 3 && verbose && initialize
    fprintf('] %5.0f seconds\n',round( etime(clock, tStart) ));
end

end

