function [ par, h_parset] = generate_parameters( h_layout, overlap, usage, check_parfiles )
%GENERATE_PARAMETERS Generates large scale parameters and stores them in track objects
%
% Normally, parameters are handled by objects of the "parameter_set" class,
% which are generated by calling "layout.create_parameter_sets". Those
% objects then feed the parameters to the "channel_builder". However, this
% method is rather inflexible when the user wants to manipulate the
% parameters directly. As an alternative, parameters can be provided in the
% property "track.par" of the "track" class. This allows the user to edit
% parameters without dealing with the "parameter_set" objects.
%
% GENERATE_PARAMETERS extracts the large scale parameters (LSPs) for the
% given scenario from the "parameter_set" class and stores them in
% "track.par". Hence, it automatically generates the LSPs and, thus,
% implements an easy-to-use interface for the "parameter_set" class.
%
% For example, a linear track with 3 segments might be given by:
%
%    t = track('linear',30);
%    t.interpolate_positions(10);
%    t.segment_index = [ 1 , 120  , 260 ];
%    t.scenario = {'BERLIN_UMa_LOS','BERLIN_UMa_NLOS','BERLIN_UMa_LOS'};
%
% This track is put in a layout with two transmitters:
%
%    l = layout;
%    l.track = t;
%    l.no_tx = 2;
%    l.tx_position = [ -50,50 ; 0,0 ; 25,25 ];
%
% The LSPs for both transmitters are generated by:
%
%    l.generate_parameters;
%
% The results are stored in "t.par". For example, the effective path gain
% (i.e. distance-dependent path gain together with shadow fading) can be
% plotted by:
%
%    plot( t.par.pg' );
%
%
% Options:
%
%    overlap = 0 ... 1 (default: 0.5)
%    When there are scenario transitions, KF and PG change smoothly during
%    a predefined interval. The length of that interval is a percentage of
%    previous segment. The parameter overlap adjusts this percentage,
%    ranging from 0 (i.e. very hard, step-like change at the scenario
%    boundary) to 1 (very smooth, but long transition).
%
%    usage = 0
%    Deletes all existing parameters from the tracks in the layout.
%
%    usage = 1
%    Deletes all existing parameters from the tracks in the layout and
%    generates new ones. Existing LSPs will be overwritten.
%
%    usage = 2 (default)
%    Keeps existing parameters, but generates missing ones. This is useful
%    when, for example, the effective path gain (pg) is provided, but the
%    other LSPs are unknown. In this case, the unknown "gaps" are filled
%    with values which are generated from the provided scenario
%    description.
%
%    check_parfiles = 0 / 1 (default: 1)
%    Disables (0) or enables (1) the parsing of shortnames and the
%    validity-check for the config-files. This is useful, if you know that
%    the parameters in the files are valid. In this case, this saves
%    execution time.
%
%
% QuaDRiGa Copyright (C) 2011-2015 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


% Parse the input variables
if exist( 'overlap' , 'var' ) && ~isempty( overlap )
    if ~( isnumeric(overlap) && all(size(overlap) == [1 1]) && isreal(overlap) ...
            && overlap<=1 && overlap>=0 )
        error('??? "overlap" must be scalar, and in between 0 and 1')
    end
else
    overlap = 0.5;
end

if exist( 'usage' , 'var' ) && ~isempty( usage )
    usage = real( usage(1) );
else
    usage = 2;
end

if exist( 'check_parfiles' , 'var' ) && ~isempty( check_parfiles )
    if ~( all(size(check_parfiles) == [1 1]) ...
            && (isnumeric(check_parfiles) || islogical(check_parfiles)) ...
            && any( check_parfiles == [0 1] ) )
        error('??? "check_parfiles" must be 0 or 1')
    end
else
    check_parfiles = true;
end

% Reset the pairing matrix. We calculate a set of parameters for each track
% and for each Tx in the layout, regardless of it being needed or not.
pairing = h_layout.pairing;
h_layout.set_pairing;

% Generate large scale parameters using the correlation maps
switch usage
    case 0  % Delete all existing parameters
        for n = 1:h_layout.no_rx
            h_layout.track(n).par_nocheck = [];
        end
        
    case 1  % Overwrite existing parameters
        h_parset = h_layout.create_parameter_sets( 0 , check_parfiles);
        h_parset.update_parameters( 1 );   % Force new parameters
        
    case 2  % Generate only non-existing parameters
        h_parset = h_layout.create_parameter_sets( 0 , check_parfiles);
        h_parset.update_parameters( 0 );
        
    otherwise
        error('??? "usage" must be 0, 1 or 2');
end

if usage > 0
    
    % Replace the scenario shortnames with their long form
    [ sup_scenarios , file_names ] = ...
        parameter_set.supported_scenarios(1);
    
    % Disable per-antenna SF and KF processing. This would create one vector of
    % SF- and KF parameters for each Rx-antenna. However, the manual parameter
    % interface currently supports only one vector for each Rx-track.
    drifting_precision = h_layout.simpar.drifting_precision;
    if drifting_precision == 3;
        h_layout.simpar.drifting_precision = 1;
    end
    
    % Extract the positions from h_parset.
    % Those are needed for matching the parameters to segments.
    pos = cell( size(h_parset) );
    o_parset = cell( size(h_parset) );
    for n = 1:numel( h_parset )
        pos{n} = h_parset(n).positions;
        o_parset{n} = ones(1,h_parset(n).no_positions);
    end
    
    % Extract the list of scenarios from h_parset.
    scenarios = { h_parset(:,1).scenario };
    
    % Calculate the number of tracks( n_rx), the number of transmitters (n_tx).
    n_trk = h_layout.no_rx;
    n_tx = h_layout.no_tx;
    par = cell(1,n_trk);
    
    % To the calculations for each Rx (i.e. for each track in the layout.)
    for i_trk = 1 : n_trk
        
        % Extract tracks and subtracks for faster processing.
        trk = h_layout.track(i_trk);
        subtrk = trk.get_subtrack;
        
        % Some temporary variables
        n_seg  = trk.no_segments;
        n_snap = trk.no_snapshots;
        z_seg  = zeros( n_tx, n_seg );
        z_snap = zeros( n_tx, n_snap );
        
        data = struct('ds',z_seg,'kf',z_snap,'pg',z_snap,...
            'asD',z_seg,'asA',z_seg,'esD',z_seg,'esA',z_seg,'xpr',z_seg);
        
        % Read scenario ids from the track
        scen = trk.scenario;
        if size(scen,1) < n_tx
            scen( 2:n_tx,: ) = scen( ones(1,n_tx-1) , : );
        end
        pos_trk = trk.positions_abs;
        
        % Replace the scenario shortnames with their long form
        for i_scen = 1:numel( scen )
            ind = strcmp( scen(i_scen) , sup_scenarios );
            if any(ind)
                scen{i_scen} = file_names{ind}(1:end-5);
            else
                i1 = regexp( scen{i_scen} , '.conf', 'once' );
                i2 = regexp( scen{i_scen} , filesep);
                
                if ~isempty( i1 )
                    if isempty( i2 )
                        scen{i_scen} = scen{i_scen}(1:i1-1);
                    else
                        scen{i_scen} = scen{i_scen}(i2(end)+1:i1-1);
                    end
                end
            end
        end
        
        % Check if the track is closed.
        closed = trk.closed;
        
        % Placeholder for the KF and the PG
        pg = cell( n_tx,n_seg );
        kf = cell( n_tx,n_seg );
        
        % Read the initial segment positions
        pos_seg = zeros( 3, n_seg );
        for i_seg = 1:n_seg
            pos_seg(:,i_seg) = pos_trk( : , trk.segment_index( i_seg ) );
        end
        
        % Do for each segment of the track
        for i_seg = 1 : n_seg
            
            % === Determine the overlapping area ===
            
            % This determines the the start and end-segments for closed tracks
            domerge = true;
            if i_seg < n_seg
                i_seg2 = i_seg + 1;
            elseif closed
                i_seg2 = 1 ;
            else
                i_seg2 = [];
                domerge = false;
            end
            
            % Determine the start and end-points of overlapping and exclusive
            % parts.
            if domerge
                rangem = zeros(3,2);   % The overlapping start / end points
                rangec = zeros(2,2);   % The non-overlapping start / end points
                
                % Determine the end index of the second segment
                rangem(2,2) = subtrk( i_seg2 ).segment_index(end)-1;
                
                % Determine the start index of the second segment
                rangem(2,1) = floor( rangem(2,2) - (rangem(2,2)-2) * overlap );
                
                % Number of overlapping snapshots
                S = rangem(2,2) - rangem(2,1) + 1;
                
                % Determine the end index of the first segment
                rangem(1,2) = subtrk( i_seg ).no_snapshots;
                
                % Determine the start index of the first segment
                rangem(1,1) = rangem(1,2) - S + 1;
                
                % Determine the the non-overlapping area of the first segment
                rangec(1,:) = [ subtrk(i_seg).segment_index(end) ,rangem(1,1)-1 ];
                
                % Determine the indexes of the non-overlapping part in the output channel
                if i_seg == 1
                    rangec(2,1) = 1;
                else
                    segments_before = 1:(i_seg-1);
                    
                    tmp = zeros( numel(segments_before),1 );
                    for i_tmp = 1:numel(segments_before)
                        tmp( i_tmp ) = subtrk( segments_before(i_tmp) ).segment_index(end);
                    end
                    
                    rangec(2,1) = sum( cat(1,subtrk( segments_before ).no_snapshots) -tmp+1 ) + 1;
                end
                rangec(2,2) = rangec(1,2) - rangec(1,1) + rangec(2,1);
                
                % Determine the indexes of the overlapping part in the output channel
                rangem(3,:) = [rangec(2,2)+1, rangec(2,2)+S];
                
                % Extend the range of the data structure
                rangem = [rangem(1,1):rangem(1,2) ; rangem(2,1):rangem(2,2) ; rangem(3,1):rangem(3,2) ];
                rangec = [rangec(1,1):rangec(1,2) ; rangec(2,1):rangec(2,2)];
                
                % Calculate the ramp
                tmp = (1:S)'/(S+1);
                ramp = 0.5*(1+sin((tmp-0.5)*pi));
                
            else
                % Here, we only have an exclusive part without overlapping
                rangec = zeros(2,2);   % The non-overlapping start / end points
                
                % Determine the the non-overlapping area of the segment
                rangec(1,:) = [ subtrk(i_seg).segment_index(end) ,...
                    subtrk(i_seg).no_snapshots  ];
                
                if i_seg > 1
                    segments_before = 1:(i_seg-1);
                    
                    tmp = zeros( numel(segments_before),1 );
                    for i_tmp = 1:numel(segments_before)
                        tmp( i_tmp ) = subtrk( segments_before(i_tmp) ).segment_index(end);
                    end
                    
                    rangec(2,1) = sum( cat(1,subtrk( segments_before ).no_snapshots) -tmp+1 ) + 1;
                    rangec(2,2) = rangec(1,2) - rangec(1,1) + rangec(2,1);
                else
                    rangec(2,1) = rangec(1,1);
                    rangec(2,2) = rangec(1,2);
                end
                
                % Extend the range of the data structure
                rangec = [rangec(1,1):rangec(1,2) ; rangec(2,1):rangec(2,2)];
            end
            % === End ===
            
            % Do for each transmitter
            % Extract the initial parameters from the parameter_sets
            for i_tx = 1 : n_tx
                
                % Determine the index of the current segment in the
                % parameter_set objects.
                i_scen = strcmp( scen( i_tx , i_seg ) , scenarios );
                tmp = pos{ i_scen , i_tx } - pos_seg( : , i_seg*o_parset{ i_scen , i_tx } );
                tmp = sum( tmp.^2 , 1 );
                [~,i_par] = min( tmp );
                
                % Extract the SF and KF traces for the current segment
                if isempty( pg{i_tx,i_seg} )
                    evaltrack = h_parset( i_scen,i_tx ).rx_track( i_par );
                    if usage == 1
                        evaltrack.par = [];
                    end
                    [ tmp_sf, tmp_kf ] = h_parset( i_scen,i_tx ).get_sf_profile( evaltrack );
                    
                    if isempty(evaltrack.par) || isempty( evaltrack.par.pg )
                        [ tmp_loss , scale_sf ] = h_parset( i_scen,i_tx ).get_pl( evaltrack );
                    else
                        tmp_loss = 0;
                        scale_sf = 1;
                    end
                    
                    % Merging of channel coefficient id done using the L1-Norm.
                    % We have to do the same here.
                    pg{ i_tx, i_seg } = sqrt( 10.^( 0.1*( -tmp_loss +...
                        10*log10( tmp_sf ) .* scale_sf )));
                    kf{ i_tx, i_seg } = sqrt( tmp_kf );
                end
                
                % Determine the SF and KF traces for the second segment in the
                % overlapping area.
                if domerge
                    i_scen2 = strcmp( scen( i_tx , i_seg2 ) , scenarios );
                    tmp = pos{ i_scen2 , i_tx } -...
                        pos_seg(:,i_seg2*o_parset{ i_scen2 , i_tx } );
                    tmp = sum( tmp.^2 , 1 );
                    [~,i_par2] = min( tmp );
                    
                    if isempty( pg{i_tx,i_seg2} )
                        evaltrack = h_parset( i_scen2,i_tx ).rx_track( i_par2 );
                        if usage == 1
                            evaltrack.par = [];
                        end
                        [ tmp_sf, tmp_kf ] = h_parset( i_scen2,i_tx ).get_sf_profile( evaltrack );
                        
                        if isempty(evaltrack.par) || isempty( evaltrack.par.pg )
                            [ tmp_loss , scale_sf ] =...
                                h_parset( i_scen2,i_tx ).get_pl( evaltrack );
                        else
                            tmp_loss = 0;
                            scale_sf = 1;
                        end
                        
                        % Merging of channel coefficients in "channel.merge" is
                        % done using the L1-Norm. We have to do the same here.
                        pg{i_tx,i_seg2} = sqrt( 10.^( 0.1*( -tmp_loss +...
                            10*log10( tmp_sf ) .* scale_sf )));
                        kf{ i_tx,i_seg2 } = sqrt( tmp_kf );
                    end
                end
                
                % Copy the segment-parameters from the parameter_set objects.
                data.ds( i_tx,i_seg )  = h_parset( i_scen,i_tx ).ds( i_par );
                data.asD( i_tx,i_seg ) = h_parset( i_scen,i_tx ).asD( i_par );
                data.asA( i_tx,i_seg ) = h_parset( i_scen,i_tx ).asA( i_par );
                data.esD( i_tx,i_seg ) = h_parset( i_scen,i_tx ).esD( i_par );
                data.esA( i_tx,i_seg ) = h_parset( i_scen,i_tx ).esA( i_par );
                data.xpr( i_tx,i_seg ) = 10*log10( h_parset( i_scen,i_tx ).xpr( i_par ) );
                
                % Extract SF and KF parts
                % This requires merging the overlapping parts from two
                % successive segments.
                
                % Copy the exclusive data
                data.pg( i_tx,rangec(2,:) ) = pg{i_tx,i_seg}( rangec(1,:) );
                data.kf( i_tx,rangec(2,:) ) = kf{i_tx,i_seg}( rangec(1,:) );
                
                % Merge overlapping parts
                if domerge
                    
                    % Merge the SF
                    A = pg{i_tx,i_seg}( rangem(1,:) );
                    B = pg{i_tx,i_seg2}( rangem(2,:) );
                    data.pg( i_tx,rangem(3,:) ) = A.*(1-ramp) + B.*ramp;
                    
                    % Merge the KF
                    A = kf{i_tx,i_seg}( rangem(1,:) );
                    B = kf{i_tx,i_seg2}( rangem(2,:) );
                    data.kf( i_tx,rangem(3,:) ) = A.*(1-ramp) + B.*ramp;
                    
                end
            end
        end
        
        % Transform the output into log-values
        data.pg = 10*log10( data.pg.^2 );
        data.kf = 10*log10( data.kf.^2 );
        
        % Processing of subtracks ignores the last value in case of a
        % closed track. This is fixed here.
        if closed
            data.pg( :,end ) = data.pg( :,1 );
            data.kf( :,end ) = data.kf( :,1 );
        end
        
        % Write the data to the track object
        par{i_trk} = data;
        h_layout.track(i_trk).par = data;
    end
    
    % Set the drifting precision back to its original value
    h_layout.simpar.drifting_precision = drifting_precision;
end

% Set the pairing matrix back to its original values
h_layout.pairing = pairing;

end
