function generate_initial_angles( h_cb )
%GENERATE INITIAL_ANGLES Generate angular parameters (private)
%
% In three dimensional space, the signals originating from a cluster would
% arrive at the receiver under a certain angle in azimuth and elevation
% direction. The same hold for the angles of departure.
%
% Note: All generated angles are in radians!
%
% QuaDRiGa Copyright (C) 2011-2014 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

N = h_cb.par.no_positions;                   % no. positions
L = h_cb.par.scenpar.NumClusters;         	  % no. taps

oL = ones(1,L);
oN = ones(1,N);

angles = h_cb.par.get_angles.'*pi/180;         % Angles between Tx and Rx
pow = h_cb.pow;                              % Path powers

spreads = [ h_cb.par.asD ; h_cb.par.asA ; ...
    h_cb.par.esD ; h_cb.par.esA ].' * pi/180;

drifting_precision = h_cb.par.simpar.drifting_precision;


if drifting_precision == 4
    % We only need arrival angles here
    use_spreads = [2,4];
else
    use_spreads = 1:4;
end

% Azimuth spreads
if L > 1
    for i_spread = use_spreads
        
        as    = spreads(:,i_spread);
        sigma = zeros( N,L );
        aso   = zeros( N,1 );
        idx   = true(N,1);
        
        cnt = 1;
        while any( idx ) && cnt < 20
            
            sigma( idx,: ) = as(idx,oL);
            sigma( idx,: ) = sigma( idx,: ) .* randn(sum( idx ),L);
            sigma( idx,1 ) = 0;
            
            if i_spread > 2 % Elevation angles
                % For the elevation angles, we have to include the LOS
                % elevation.
                sigma = sigma + angles(:,i_spread*oL);
            end
            
            tmp = sigma( idx,: );
            for n = 1:5
                
                % Map angels to the allowed range if they are outside
                tmp = mod( tmp + pi, 2*pi) - pi;
                if i_spread > 2 % Elevation angles
                    tmp( tmp >  pi/2 ) =  pi - tmp( tmp >  pi/2 );
                    tmp( tmp < -pi/2 ) = -pi - tmp( tmp < -pi/2 );
                end
                
                aso( idx,: ) = calc_angular_spreads( tmp, pow( idx,: ) );
                scale        = as( idx,: )./aso( idx,: );
                
                % Determine the angles that are outside the possible range
                if i_spread > 2 % Elevation angles
                    
                    tmp      = scale(:,oL) .* (tmp - angles(idx,i_spread*oL)) +...
                        angles(idx,i_spread*oL);
                    
                    out      = tmp > pi/2;
                    tmp(out) = randn( sum(out(:)) , 1 ) * pi/4 + pi/2;
                    
                    out      = tmp < -pi/2;
                    tmp(out) = randn( sum(out(:)) , 1 ) * pi/4 - pi/2;
                    
                else % Azimuth angles
                    
                    tmp      = scale(:,oL) .* tmp;
                    out      = tmp > pi | tmp < -pi;
                    tmp(out) = randn( sum(out(:)) , 1 ) * pi/2 + pi;
                    
                end
            end
            
            % Map angels to the allowed range if they are outside
            tmp = mod( tmp + pi, 2*pi) - pi;
            if i_spread > 2 % Elevation angles
                tmp( tmp >  pi/2 ) =  pi - tmp( tmp >  pi/2 );
                tmp( tmp < -pi/2 ) = -pi - tmp( tmp < -pi/2 );
            end
            sigma( idx,: ) = tmp;
            
            aso( idx,: )   = calc_angular_spreads( sigma( idx,: ), pow( idx,: ) );
            
            if cnt == 1
                sigma_old = sigma;
                aso_old   = aso;
            else
                id_better = aso > aso_old;
                id_worse  = aso < aso_old;
                
                sigma_old( id_better,: ) = sigma( id_better,: );
                sigma( id_worse,: )      = sigma_old( id_worse,: );
                
                aso_old( id_better ) = aso( id_better );
                aso( id_worse )      = aso_old( id_worse );
            end
            
            idx = aso < 0.95*as | aso > 1.05*as;
            cnt = cnt + 1;
        end
        
        if i_spread <= 2 % Azimuth angles
            sigma = sigma + angles(:,i_spread*oL);
            sigma = mod( sigma + pi, 2*pi) - pi;
        end
        
        switch i_spread
            case 1
                h_cb.AoD = sigma;
            case 2
                h_cb.AoA = sigma;
            case 3
                h_cb.EoD = sigma;
            case 4
                h_cb.EoA = sigma;
        end
    end
else
    h_cb.AoD = angles(:,1);
    h_cb.AoA = angles(:,2);
    h_cb.EoD = angles(:,3);
    h_cb.EoA = angles(:,4);
end

if drifting_precision == 4
    % In this mode, the departure angles depend on the arrival angles.
    % Hence, we calculate the corresponding departure angles.
    
    % Calculate the unit length vector pointing from the Rx to the LBS.
    [ ahat_x, ahat_y, ahat_z ] = sph2cart( h_cb.AoA, h_cb.EoA, 1);
    ahat = permute( cat( 3, ahat_x, ahat_y, ahat_z ) , [3,1,2] );
    
    r = cat( 2,h_cb.par.rx_track.initial_position ) - h_cb.par.tx_position(:,oN);
    norm_r = sqrt(sum(r.^2)).';
    dist = h_cb.taus.*simulation_parameters.speed_of_light + norm_r(:,oL);
    
    [ b, norm_b ]  = solve_cos_theorem( ahat , r , dist  );
    
    h_cb.AoD = permute( atan2(b(2,:,:),b(1,:,:) ) , [2,3,1]);
    h_cb.EoD = permute( asin( b(3,:,:)./norm_b  ) , [2,3,1]);
    
    
elseif drifting_precision == 5
    % This implements the multi-bounce model.
    % Those values are constant and need to be calculated only once
    
    [ ahat_x, ahat_y, ahat_z ] = sph2cart(h_cb.AoA, h_cb.EoA, 1);
    ahat = permute( cat( 3, ahat_x, ahat_y, ahat_z ) , [3,1,2] );
    ahat = reshape( ahat , 3 , N*L );
    
    [ bhat_x, bhat_y, bhat_z ] = sph2cart(h_cb.AoD, h_cb.EoD, 1);
    bhat = permute( cat( 3, bhat_x, bhat_y, bhat_z ) , [3,1,2] );
    bhat = reshape( bhat , 3 , N*L );
    
    r = cat( 2,h_cb.par.rx_track.initial_position ) - h_cb.par.tx_position(:,oN);
    norm_r = sqrt(sum(r.^2)).';
    r = reshape( r(:,:,oL) , 3 , N*L );
    
    dist = h_cb.taus.*simulation_parameters.speed_of_light + norm_r(:,oL);
    dist = permute( dist , [3,1,2] );
    dist = reshape( dist , 1 , N*L );

    % Test if the optimization problem can be solved and update the angles
    % (single-bounce) that violate the assumptions.
    [ ~, ~, ~, valid ] = solve_multi_bounce_opti( ahat, bhat, r, dist, 2 );
    invalid = reshape( ~valid, N,L ) ;
    invalid( :,1 ) = false;  % LOS is always valid
    
    % Use the single-bounce model for points where the problem has no
    % solution.
    [ b, norm_b ] = solve_cos_theorem( ahat(:,invalid(:)),...
        r(:,invalid(:)), dist(invalid(:)));
    h_cb.AoD(invalid) = atan2(b(2,:,:),b(1,:,:) );
    h_cb.EoD(invalid) = asin( b(3,:,:)./norm_b  );
    
    % Here we rescale the angles in order to maintain the angular spreads.
    rescale = find( sum( invalid ,2 ) > 0 );
    
    if any( rescale )
        % AoD rescaling
        AoD = rescale_departure_angles( spreads(rescale,1),...
            h_cb.AoD(rescale,:), pow(rescale,:),...
            [false(numel(rescale),1), ~invalid(rescale,2:end) ]);
        AoD = mod( AoD + pi, 2*pi) - pi;
        
        tmp = spreads(rescale,1) - calc_angular_spreads( AoD, pow(rescale,:) );
        ind = abs(tmp) < 1e-4;
        h_cb.AoD( rescale(ind),: ) = AoD( ind,: );
        
        % EoD rescaling
        EoD = rescale_departure_angles( spreads(rescale,3),...
            h_cb.EoD(rescale,:), pow(rescale,:),...
            [false(numel(rescale),1), ~invalid(rescale,2:end) ]);
        
        EoD = mod( EoD + pi, 2*pi) - pi;
        EoD( EoD >  pi/2 ) =  pi - EoD( EoD >  pi/2 );
        EoD( EoD < -pi/2 ) = -pi - EoD( EoD < -pi/2 );
        
        tmp = spreads(rescale,3) - calc_angular_spreads( EoD, pow(rescale,:) );
        ind = abs(tmp) < 1e-4;
        h_cb.EoD( rescale(ind),: ) = EoD( ind,: );
    end
end

end
