function chan_out = split_tx( h_channel, varargin )
%SPLIT_TX Splits channel arrays based on transmit antenna indices
%   
%   This function can be used to split large transmit antenna arrays into
%   smaller arrays. For example, this can be used to calculate the
%   channels for individual sectors at a BS.
%
%   Example:
%   A channel array has channels from three base stations (BSs). The first
%   and second BS have two sectors, each with two antennas. However, the
%   sector antennas are merged into one array. The third BS has only one
%   sector. To split the channels into five sectors, the following command
%   can be used:    
% 
%       cs = c.split( {1:2,3:4}, {1:2,3:4}, {1:2} );
%
%   Notes: 
%   - The order of the inputs must match the transmitters in alphabetical
%     order, i.e. the first input corresponds to "Tx01", the second to "Tx02"
%     and so on. This is independent of the order in "layout.tx_name", which
%     might have a different order.   
%   - Outputs are alphabetically sorted.
%
%
% QuaDRiGa Copyright (C) 2011-2015 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.

splt = varargin;

% The number of channel objects
no_c = numel( h_channel );

% Parse names of the channel objects
id_tx = cell(no_c,1);
id_rx = cell(no_c,1);
id_trk = cell(no_c,1);
for n = 1 : no_c
    name = h_channel(n).name;
    p = regexp( name , '_' );
    id_tx{n} = name(1:p(1)-1);
    id_rx{n} = name(p(1)+1:end);
    id_trk{n} = name;
end

% Get the unique tx_names
% The following code implements "unique(id_trk)" but does not sort the
% values in id_trk.

tmp = id_tx;
pt = 1;
while pt < numel(tmp)
    ii = find( strcmp( tmp(pt) , tmp(pt+1:end) ) )+pt;
    tmp = tmp( setdiff( 1:numel(tmp), ii ) );
    pt  = pt + 1;
end
tx_name = tmp;
no_bs = numel( tx_name );
tx_name = sort( tx_name );

% Parse input data
if numel( splt ) ~= no_bs
   if  numel( splt ) == 1
       if ~iscell( splt{1} )
           error('??? Inputs must be a cell array.');
       else
           % Copy the data
           for n = 2:no_bs
               splt{n} = splt{1};
           end
       end
   else
        error('??? Number of inputs does not match the number of BSs in the channel array.');
   end
end

for n = 1:numel( splt )
    for m = 1:numel( splt{n} )
        if size( splt{n}{m},1 ) ~= 1
            splt{n}{m} = splt{n}{m}(:)';
        end
        if any( rem( splt{n}{m} , 1 ) ~= 0 )
            error('??? Inputs must be vectors of integer numbers.');
        end
    end
end

chan_out = echannel.empty;
sec_cnt  = 1;
for i_bs = 1 : no_bs
    no_sec = numel( splt{i_bs} );
    
    % Do for each input channel
    for i_c = 1 : no_c
        
        % Check if it matches the input data structure
        if strcmp( id_tx( i_c ), tx_name( i_bs ) )
            
            for i_sec = 1 : no_sec
                tx_ind = splt{i_bs}{i_sec};
            
                ii   = regexp( h_channel(i_c).name , '_' );
                name = [ h_channel(i_c).name(1:ii(1)-1),'s',num2str(i_sec),...
                    h_channel(i_c).name(ii:end) ];
                
                tmp = channel;
                tmp.name  = name;
                tmp.version = h_channel(i_c).version;
                tmp.individual_delays = h_channel(i_c).individual_delays;
                tmp.coeff = h_channel(i_c).coeff( :,tx_ind,:,: );
                
                if h_channel(i_c).individual_delays
                      tmp.delay = h_channel(i_c).delay( :,tx_ind,:,: );
                else
                    tmp.delay = h_channel(i_c).delay;
                end
                tmp.tx_position = h_channel(i_c).tx_position;
                tmp.rx_position = h_channel(i_c).rx_position;
                
                chan_out(sec_cnt) = tmp;
                sec_cnt = sec_cnt + 1;
            end
        end
    end
end

name = cell( numel( chan_out ),1 );
for n = 1 : numel( chan_out )
    name{n} = chan_out(n).name;
end
[~,ind] = sort( name );
chan_out = chan_out(ind);

end
