%% Network Setup and Parameter Generation
%
% The tutorial demonstrates how to setup a simple layout with multiple receivers, how to adjust
% parameters manually, generate channel coefficients, and how to calculate simple parameters from
% the data. The channel model class 'qd_builder' generates correlated values for the LSPs. The
% channel builder then uses those values to create coefficients that have the specific properties
% defined in the builder objects. One important question is therefore: Can the same properties which
% are defined in the builder also be found in the generated coefficients? This is an important test
% to verify, if all components of the channel builder work correctly.

%% Channel model setup and coefficient generation
% We first set up the basic parameters.

close all
clear all

set(0,'defaultTextFontSize', 18)                        % Default Font Size
set(0,'defaultAxesFontSize', 18)                        % Default Font Size
set(0,'defaultAxesFontName','Times')                    % Default Font Type
set(0,'defaultTextFontName','Times')                    % Default Font Type
set(0,'defaultFigurePaperPositionMode','auto')          % Default Plot position
set(0,'DefaultFigurePaperType','<custom>')              % Default Paper Type

s = qd_simulation_parameters;                           % Set up simulation parameters
s.show_progress_bars = 0;                               % Disable progress bars
s.center_frequency = 2.53e9;                            % Set center frequency
s.sample_density = 2;                                   % 2 samples per half-wavelength
s.use_absolute_delays = 1;                              % Include delay of the LOS path

%%
% We have one transmitter and 250 receiver positions. Each receiver gets a specific channel.
% However, the receivers LSPs will be correlated. We use omni directional antennas at all terminals.

l = qd_layout(s);                                       % Create new QuaDRiGa layout
l.no_rx = 250;                                          % Set 250 MTs
l.randomize_rx_positions( 200 , 1.5 , 1.5 , 1 );        % 200 m radius, 1.5 m Rx height
l.set_scenario('BERLIN_UMa_NLOS');                      % Use NLOS scenario

l.tx_position(3) = 25;                                  % 25 m tx height
l.tx_array = qd_arrayant( 'omni' );                     % Omnidirectional BS antenna
l.rx_array = qd_arrayant( 'omni' );                     % Omnidirectional MT antenna

set(0,'DefaultFigurePaperSize',[14.5 7.3])              % Adjust paper size for plot
l.visualize([],[],0);                                   % Plot the layout
view(-33, 60);                                          % Enable 3D view

%%
% We set up the scenario such that there is no XPR. I.e. all vertical polarized paths will remain
% vertical after a reflection. The same result would be achieved with a perfectly X-polarized array
% antenna at the receiver and summing up the power over all elements. We further increase the KF to
% have a wider spread. This allows us to study the parameters at a wider range when evaluating the
% results.

p = l.init_builder;                                     % Initialize builder
p.plpar = [];                                           % Disable path-loss
p.scenpar.XPR_mu    = 100;                              % Disable XPR
p.scenpar.XPR_sigma = 0;
p.scenpar.KF_mu     = 5;                                % Increase KF-Range
p.scenpar.KF_sigma  = 15;
p.scenpar.DS_mu     = log10(0.6e-6);                    % Median DS = 600 ns
p.scenpar.DS_sigma  = 0.3;                              % 300-1200 ns range

p.gen_ssf_parameters;                                   % Generate small-scale-fading
c = p.get_channels;                                     % Generate channels

coeff = squeeze( cat( 1, c.coeff ) );                   % Extract amplitudes and phases
delay = permute( cat(1,c.delay) , [1,3,4,2] );          % Extract path delays

%% Results and discussion
% In the following four plots, we extract parameters from the generated coefficients and compare
% them with the initial ones which were generated by the 'parameter_set' object (P). The values in
% (P) can be seen as a request to the channel builder and the values in the generated coefficients
% (C) as a delivery. We first calculate the SF from the channel data by summing up the power over
% all 20 taps. We see, that the values are almost identical.

sf = sum(mean( abs(coeff).^2 ,3),2);                    % Calculate shadow fading

set(0,'DefaultFigurePaperSize',[14.5 4.5])              % Change Paper Size
figure('Position',[ 100 , 100 , 760 , 400]);            % New figure

plot(-35:35,-35:35,'k')
hold on
plot([-35:35]+3,-35:35,'--k')
plot([-35:35]-3,-35:35,'--k')
plot( 10*log10(p.sf') , 10*log10(sf) , '.'  )
hold off
axis([ -25 , 25 , -25, 25 ])
legend('Equal','+/- 3dB','Location','SouthEast')
xlabel('SF_P [dB]'); ylabel('SF_C [dB]');
title('Shadow Fading - Requested vs. generated value');

%%
% Next, we repeat the same calculation for the K-Factor. Again, we see that the values are almost identical.

p_nlos = sum(mean( abs(coeff(:,2:end,:)).^2 ,3),2);     % Calculate NLOS power
p_los  = mean( abs(coeff(:,1,:)).^2 ,3);                % Calculate LOS power
kf = p_los./p_nlos;                                     % Calculate K-Factor

figure('Position',[ 100 , 100 , 760 , 400]);            % New figure
plot(-35:35,-35:35,'k')
hold on
plot([-35:35]+3,-35:35,'--k')
plot([-35:35]-3,-35:35,'--k')
plot( 10*log10(p.kf') , 10*log10(kf) , '.'  )
hold off
axis([ -30 , 30 , -30, 30 ])
legend('Equal','+/- 3dB','Location','SouthEast')
xlabel('KF_P [dB]');
ylabel('KF_C [dB]');
title('K-Factor - Requested vs. generated value');

%%
% Now we repeat the calculation for the RMS delays spread.

pow_tap = abs(coeff).^2;                                % Calculate path powers
pow_sum = sum(pow_tap,2);                               % Calculate sum-power
mean_delay = sum( pow_tap.*delay,2) ./ pow_sum;         % Calculate mean delay
ds = sqrt( sum( pow_tap.*delay.^2 ,2)./ pow_sum - mean_delay.^2 );
ds = mean(ds,3);                                        % Calculate delay spread

figure('Position',[ 100 , 100 , 760 , 400]);            % New figure
plot([0:0.1:2],[0:0.1:2],'k')
hold on
plot([0:0.1:2]*1.1,[0:0.1:2],'--k')
plot([0:0.1:2],[0:0.1:2]*1.1,'--k')
plot( p.ds'*1e6 , (ds')*1e6 , '.'  )
hold off
axis([ 0,1.5,0,1.5 ])
legend('Equal','+/- 10% Error','Location','SouthEast')
xlabel('DS_P [\mus]');
ylabel('DS_C [\mus]');
title('Delay Spread - Requested vs. generated value');
