classdef qd_sos < handle
%QD_SOS QuaDRiGa sum-of-sinuoids log-normal autocorrelation model
%
% DESCRIPTION
% 
% This class implements the sum-of-sinusoids method for generating spatially correlated random variables. An
% autocorrelation function (ACF), i.e. a description of the correlation vs. distance, needs to be provided. This
% function is approximated by a fourier series. The coefficients of the series can the be used to generate spatially
% correlated random variables.
%
% The deafult ACF is defined by a combination of the Gauss profile (ranging from 0 to the decorrelation distance) and a
% exponential profile (ranging from the decorrelation distance to a maximum distance of 5 time the decorrelation
% distance. It is approximated by 400 sinusoids in 3D coordinates.
%
% Input (constructor):
%   dist_decorr     Decorrelation distance in [m], Default: 10 m
%   distribution    Distribution of the random variables, Default: Normal
%
% QuaDRiGa Copyright (C) 2011-2017 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% Fraunhofer Heinrich Hertz Institute
% Wireless Communication and Networks
% Einsteinufer 37, 10587 Berlin, Germany
%
% This file is part of QuaDRiGa.
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% QuaDRiGa is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%
% You should have received a copy of the GNU Lesser General Public License
% along with QuaDRiGa. If not, see <http://www.gnu.org/licenses/>.

    properties
        % Name of the preset
        name = 'Custom';
        
        % Distribution of random variables (Normal or Uniform)
        distribution = 'Normal';
    end
    
    properties(Dependent)
        % Decorrelation distance in [m]
        dist_decorr
        
        % The number of dimensions (1, 2 or 3)
        dimensions
        
        % Number of sampling frequencies
        no_coefficients
    end
    
    properties(SetAccess=protected)
        % Vector of sample points for the ACF in [m]
        dist = [];
        
        % Desired aotocorrelation function
        acf = [];
        
        % Sinusoid coefficients (frequencies) 
        sos_freq = [];
        
        % Phase offsets
        sos_phase = [];
        
        % Amplitude of the sinusoid coefficients
        sos_amp = [];
        
        % Mean-square error of the approximation. 
        %   The first value only contains only the values from 0 to the maximum distance for which the ASC is specified.
        %   The second values also covers the values outside the specified range (which are assumed to be 0 in the
        %   desired ACF).
        approx_mse
    end
    
    properties(Access=private)
        Pdist_decorr = [];
    end
    
    properties(Hidden)
        OctEq = false; % For qf.eq_octave
    end
    
    methods
        % Constructor
        function h_sos = qd_sos( dist_decorr, distribution )
            if ~exist( 'dist_decorr','var' ) || (exist( 'dist_decorr','var' ) && ~isempty( dist_decorr ) )
                
                R =  [1, 0.999375, 0.997503, 0.994391, 0.99005, 0.984496, 0.977751, 0.969839, 0.960789, 0.950634, 0.939412, 0.927162,...
                    0.91393, 0.89976, 0.884704, 0.868813, 0.852141, 0.834745, 0.816683, 0.798014, 0.778797, 0.759093, 0.738964,...
                    0.71847, 0.697671, 0.676629, 0.655401, 0.634046, 0.61262, 0.59118, 0.569776, 0.548462, 0.527286, 0.506294,...
                    0.48553, 0.465036, 0.444851, 0.42501, 0.405547, 0.386492, 0.367876, 0.358793, 0.349934, 0.341294, 0.332867,...
                    0.324649, 0.316633, 0.308815, 0.301191, 0.293754, 0.286501, 0.279427, 0.272528, 0.265799, 0.259237, 0.252836,...
                    0.246594, 0.240505, 0.234567, 0.228775, 0.223127, 0.217618, 0.212245, 0.207004, 0.201893, 0.196908, 0.192047,...
                    0.187305, 0.18268, 0.17817, 0.173771, 0.16948, 0.165296, 0.161215, 0.157234, 0.153352, 0.149566, 0.145873,...
                    0.142271, 0.138759, 0.135333, 0.131991, 0.128732, 0.125554, 0.122454, 0.11943, 0.116482, 0.113606, 0.110801,...
                    0.108065, 0.105397, 0.102795, 0.100257, 0.0977812, 0.0953669, 0.0930123, 0.0907158, 0.088476, 0.0862915, 0.0841609,...
                    0.0820829, 0.0800563, 0.0780797, 0.0761519, 0.0742716, 0.0724379, 0.0706493, 0.068905, 0.0672037, 0.0655444,...
                    0.0639261, 0.0623477, 0.0608084, 0.059307, 0.0578427, 0.0564145, 0.0550216, 0.0536631, 0.0523382, 0.0510459,...
                    0.0497856, 0.0485564, 0.0473575, 0.0461882, 0.0450478, 0.0439356, 0.0428508, 0.0417928, 0.0407609, 0.0397545,...
                    0.0387729, 0.0378156, 0.0368819, 0.0359713, 0.0350832, 0.034217, 0.0333721, 0.0325482, 0.0317445, 0.0309608, 0.0301963,...
                    0.0294508, 0.0287236, 0.0280144, 0.0273227, 0.0266481, 0.0259902, 0.0253485, 0.0247226, 0.0241122, 0.0235169,...
                    0.0229362, 0.0223699, 0.0218176, 0.0212789, 0.0207535, 0.0202411, 0.0197414, 0.0192539, 0.0187786, 0.0183149,...
                    0.0178627, 0.0174217, 0.0169915, 0.016572, 0.0161628, 0.0157638, 0.0153745, 0.0149949, 0.0146247, 0.0142636,...
                    0.0139115, 0.013568, 0.013233, 0.0129063, 0.0125876, 0.0122768, 0.0119737, 0.011678, 0.0113897, 0.0111085, 0.0108342,...
                    0.0105667, 0.0103058, 0.0100514, 0.0098032, 0.00956116, 0.00932509, 0.00909485, 0.00887029, 0.00865128, 0.00843768,...
                    0.00822935, 0.00802617, 0.007828, 0.00763472, 0.00744622, 0.00726237, 0.00708306, 0.00690818];
                
                D = [ 0, 0.25, 0.5, 0.75, 1, 1.25, 1.5, 1.75, 2, 2.25, 2.5, 2.75, 3, 3.25, 3.5, 3.75, 4, 4.25, 4.5, 4.75, 5, 5.25, 5.5,...
                    5.75, 6, 6.25, 6.5, 6.75, 7, 7.25, 7.5, 7.75, 8, 8.25, 8.5, 8.75, 9, 9.25, 9.5, 9.75, 10, 10.25, 10.5, 10.75, 11, 11.25,...
                    11.5, 11.75, 12, 12.25, 12.5, 12.75, 13, 13.25, 13.5, 13.75, 14, 14.25, 14.5, 14.75, 15, 15.25, 15.5, 15.75, 16, 16.25,...
                    16.5, 16.75, 17, 17.25, 17.5, 17.75, 18, 18.25, 18.5, 18.75, 19, 19.25, 19.5, 19.75, 20, 20.25, 20.5, 20.75, 21, 21.25,...
                    21.5, 21.75, 22, 22.25, 22.5, 22.75, 23, 23.25, 23.5, 23.75, 24, 24.25, 24.5, 24.75, 25, 25.25, 25.5, 25.75, 26, 26.25,...
                    26.5, 26.75, 27, 27.25, 27.5, 27.75, 28, 28.25, 28.5, 28.75, 29, 29.25, 29.5, 29.75, 30, 30.25, 30.5, 30.75, 31, 31.25,...
                    31.5, 31.75, 32, 32.25, 32.5, 32.75, 33, 33.25, 33.5, 33.75, 34, 34.25, 34.5, 34.75, 35, 35.25, 35.5, 35.75, 36, 36.25,...
                    36.5, 36.75, 37, 37.25, 37.5, 37.75, 38, 38.25, 38.5, 38.75, 39, 39.25, 39.5, 39.75, 40, 40.25, 40.5, 40.75, 41, 41.25,...
                    41.5, 41.75, 42, 42.25, 42.5, 42.75, 43, 43.25, 43.5, 43.75, 44, 44.25, 44.5, 44.75, 45, 45.25, 45.5, 45.75, 46, 46.25,...
                    46.5, 46.75, 47, 47.25, 47.5, 47.75, 48, 48.25, 48.5, 48.75, 49, 49.25, 49.5, 49.75 ];
                
                F = [ 0.000270732, -0.00556384, -0.00451234;...
                    -0.0012283, -0.00782959, -0.0110094;...
                    -0.00323543, 0.00824413, 0.0099217;...
                    -0.0247026, 0.015971, 0.0432296;...
                    0.0105695, 0.00938849, 0.0038808;...
                    0.00600819, 0.00623642, -0.0117468;...
                    0.0109936, -0.00474035, -0.00591582;...
                    -0.00742771, -0.00164918, -0.000716562;...
                    0.00326827, 0.00423341, 0.0121304;...
                    0.00406413, -0.00181184, 0.0112387;...
                    -0.00692979, -0.00483549, -0.00794011;...
                    0.00463467, -0.000432069, 0.0129658;...
                    0.00126671, 0.004905, 0.011438;...
                    -0.0124975, -0.00041245, -0.000365891;...
                    -0.00618144, -0.014533, -0.0118621;...
                    0.0209632, -0.00368511, -0.0218447;...
                    0.0171722, 0.0122207, 0.00333916;...
                    0.0278528, -0.00891874, 0.0437698;...
                    -0.0014903, 0.0214352, 0.015475;...
                    0.00698191, -0.00979143, 0.0280287;...
                    -0.0180562, 0.017857, 0.00832413;...
                    -0.000676709, 0.0112994, -0.0283212;...
                    0.00481702, 0.0212573, 0.0212607;...
                    0.00155305, -0.0150659, 0.0264732;...
                    -0.0243545, 0.0150446, 0.0109571;...
                    -0.00696824, 0.0169711, -0.0110362;...
                    0.0072552, -0.0177448, -0.0106682;...
                    -0.0256093, -0.0152043, -0.00657407;...
                    -0.0193393, 0.0215933, 0.00948403;...
                    -0.00987895, -0.0207445, -0.0200574;...
                    -0.0450157, -0.0167559, 0.0217244;...
                    -0.00188407, 0.00148659, 0.0064001;...
                    0.00951849, -0.0042887, -0.00520458;...
                    -0.00278257, 0.00114464, 0.00650676;...
                    -0.00253032, -0.0058885, -0.00871904;...
                    0.00840134, 0.00292395, 0.00754655;...
                    0.00540068, -0.00883467, 0.00438104;...
                    0.0100762, 0.00257036, -0.0040872;...
                    -0.00225538, -0.00193984, -0.0061583;...
                    0.002445, 0.00306405, -0.0100069;...
                    0.000853789, 0.00664115, 0.000826538;...
                    -0.00732376, -0.00810441, -0.00266335;...
                    0.00974005, 0.00168491, 0.00619474;...
                    -0.00813298, -0.00460111, 0.00683532;...
                    0.00939966, 0.000672313, -0.00756988;...
                    -0.00856581, 0.00791531, -0.000238806;...
                    0.0272879, 0.00902243, -0.00314648;...
                    -0.00690754, 0.00222998, -0.00222128;...
                    0.0211152, -0.0441655, 0.0225724;...
                    -0.00380302, -0.00485736, -0.00365156;...
                    0.0168869, -0.0452261, -0.0241877;...
                    0.0192047, -0.00862483, 0.00810815;...
                    -0.00758845, 0.00694598, 0.0033567;...
                    -0.00151315, 0.0194838, -0.0178677;...
                    -0.0122926, 0.0119551, -0.0201775;...
                    -0.0128691, 0.0180129, -7.24326e-05;...
                    0.0083055, -0.00315159, 0.0207778;...
                    -0.0167885, 0.0131013, -0.00717347;...
                    -0.0190786, 0.012105, 0.00391048;...
                    0.0185084, 0.00494547, -0.0126947;...
                    -0.020107, 0.00231842, 0.0117648;...
                    0.0154465, 0.0443624, -0.0263912;...
                    -0.0264151, 0.0152426, 0.000363375;...
                    -0.0237303, 0.0152715, 0.0122484;...
                    -0.00446979, -2.46651e-05, -0.00560457;...
                    0.00214469, -0.00350608, 0.00955244;...
                    -0.0101494, -0.00204133, 0.000981998;...
                    -2.81248e-05, -0.00466257, 0.00487613;...
                    -0.00593193, -0.0155911, -0.0164159;...
                    0.0075601, -0.0186222, -0.0111459;...
                    -0.00469501, 0.0302762, -0.0285264;...
                    -0.010235, 0.0175337, -0.0114561;...
                    -0.0183174, -0.00253686, 0.0149452;...
                    0.0102726, -0.0180346, -0.0124023;...
                    0.00266861, 0.0120975, 0.0207418;...
                    -0.00255855, -0.0384273, -0.012113;...
                    0.0170495, 0.00882965, -0.014658;...
                    -0.0125527, -0.0164751, -0.0118893;...
                    -0.00395323, 0.0376043, 0.00902199;...
                    -0.00726262, 0.00217047, 0.000405521;...
                    0.0224726, -0.00520821, -0.00716658;...
                    0.0268846, 0.0111158, 0.0256719;...
                    0.0343007, -0.0170485, 0.00662899;...
                    0.0311154, -0.0230409, 0.00386954;...
                    -0.0112964, -0.00673751, 0.0199114;...
                    0.011823, 0.0179933, -0.0102053;...
                    -0.0156767, -0.0290952, 0.0198249;...
                    -0.0111112, 0.0312949, 0.0195573;...
                    0.0194737, -0.0319558, -0.00921519;...
                    -0.0211344, 0.0318013, 0.00522767;...
                    -0.0134244, 0.0142358, -0.0313459;...
                    0.0165914, 0.000914589, -0.0328074;...
                    -0.0214634, 0.0298852, 0.00227064;...
                    0.0194293, 0.00363629, 0.013303;...
                    0.0181866, 0.00102588, 0.0318442;...
                    0.00207375, 0.0245803, -0.0271545;...
                    -0.0214416, -0.01079, 0.00333577;...
                    -0.0235229, -0.00697675, -0.00150487;...
                    -0.00111369, -0.011654, -0.0526921;...
                    0.0318611, -0.0403126, -0.017704;...
                    -0.000262185, 0.00496694, 0.00455826;...
                    -0.00494158, -0.0051306, 0.00518598;...
                    -0.018703, -0.0303936, -0.0145515;...
                    0.00167665, -0.0241507, 0.00138376;...
                    0.0162386, -0.00783373, -0.0162149;...
                    -0.00285693, 0.00813909, 0.00179716;...
                    0.00674379, -0.00619354, -0.00492977;...
                    0.0222417, -0.00612, 0.00747233;...
                    -0.00884558, 0.00379607, 0.00393532;...
                    0.0292864, -0.0120927, 0.0181136;...
                    -0.0231849, -0.0281984, -0.00362886;...
                    0.00477072, -0.00113608, -0.00736502;...
                    -0.00479564, -0.0056416, -0.00477605;...
                    0.00709498, 0.019491, 0.0304739;...
                    -0.00674974, -0.00321102, -0.00722991;...
                    0.0277445, 0.00921151, 0.0251138;...
                    -0.0183283, -0.0164051, 0.00189185;...
                    -0.000365788, 0.00501902, 0.0402364;...
                    -0.00311316, 0.0206378, -0.0327961;...
                    0.0372444, 0.0391563, -0.0062365;...
                    0.0073646, 0.0227775, -0.0305054;...
                    -0.00208699, 0.00339088, 0.00960665;...
                    0.00460793, 0.00247507, -0.0381501;...
                    0.00264708, -0.00497494, -0.00370944;...
                    0.0242369, -0.0171941, 0.0244014;...
                    -0.00448306, 0.00582083, 0.00802063;...
                    0.0192472, -0.0233107, -0.0451821;...
                    0.0530316, -0.00495944, -0.0108969;...
                    -0.0227439, -0.00470684, -0.012717;...
                    -0.014056, 0.0158448, 0.0158913;...
                    0.0193809, 0.0174335, -0.00360995;...
                    0.00378614, 0.022923, 0.0334469;...
                    -0.0178901, -0.0098074, -0.0168282;...
                    -0.0195831, -0.0133679, -0.0127082;...
                    -0.0133301, -0.0150349, -0.0201969;...
                    -0.0199864, 0.00616027, -0.0221995;...
                    0.00974749, 0.0165493, 0.0210419;...
                    0.0396708, 0.00980581, 0.00659859;...
                    -0.00141259, 0.0283123, 0.00284099;...
                    -0.0185088, -0.00773435, -0.0202302;...
                    0.0199898, 0.00272656, 0.0201154;...
                    -0.0324706, 0.0234562, 0.0100833;...
                    0.00511866, -0.0395855, -0.0124652;...
                    0.000493408, -0.0284753, 0.000754947;...
                    0.0111358, 0.00514302, -0.0024567;...
                    -0.00208387, 0.00629267, -0.010609;...
                    -0.00394623, -0.0103326, -0.00673641;...
                    -0.0115353, 0.0248979, -0.031436;...
                    0.0273585, 0.0173571, -0.0264355;...
                    0.0333898, 0.0126775, -0.0217484;...
                    0.00194674, 0.0411938, -0.00691915;...
                    0.00601645, 0.010817, -0.0256603;...
                    0.0149859, 0.00778073, 0.0229464;...
                    0.00822997, 0.0100809, 0.0253436;...
                    -0.0127663, 0.0276936, -0.000557283;...
                    0.0129068, -0.000754343, -0.000277965;...
                    0.0119948, -0.00239576, -0.00419715;...
                    -0.000594593, 0.00773438, 0.0103468;...
                    0.00520396, -0.0289747, -0.00797584;...
                    -0.00776, -0.00242826, -0.0100557;...
                    -0.0252011, -0.00169951, 0.0170952;...
                    -0.00554528, 0.0292274, -0.00672584;...
                    -0.00233977, -0.0115608, -0.00711692;...
                    0.0226178, 0.0347365, -0.00811717;...
                    -0.00459074, 0.00671435, -0.0111185;...
                    0.00867306, 0.0172249, -0.0209698;...
                    -0.0193766, 0.00379408, 0.0373558;...
                    -0.00543052, 0.0162438, 0.0381476;...
                    0.0310396, 0.0273051, 0.00628987;...
                    -0.0154468, -0.0124306, -0.0368167;...
                    -0.0304472, 0.0186999, -0.0223579;...
                    0.0342967, -0.0213057, -0.0124047;...
                    0.0312648, -0.00838843, 0.0271334;...
                    0.0101076, 0.00246311, -0.0409371;...
                    -0.00435672, 0.0115853, -0.0278752;...
                    -0.0218391, -0.00675209, -0.0201913;...
                    -0.0228197, 0.0337464, 0.0127284;...
                    -0.0259365, 0.00759783, 0.0141349;...
                    -0.026814, -0.0312736, 0.0127418;...
                    0.0411875, 0.0102939, -0.00418723;...
                    -0.0407617, -0.0119292, 0.00401084;...
                    -0.00333855, -0.0121301, -0.00561177;...
                    0.00205532, -0.0122099, 0.00694835;...
                    -0.00954671, 0.00561579, -0.0088004;...
                    -0.0264909, -0.00993241, 0.0113932;...
                    0.00620335, 0.0227824, -0.0193056;...
                    -0.0026793, 0.0134336, -0.00373454;...
                    -0.0136013, -0.00196643, 0.00356709;...
                    0.0327047, -0.0204834, -0.0261147;...
                    -0.0230231, 0.0108556, 0.039051;...
                    -0.00569886, -0.00453724, 0.0122096;...
                    -0.00931217, -0.0380417, -0.0258881;...
                    0.0088339, -0.0460924, 0.0012402;...
                    0.028877, -0.0361005, 0.0059798;...
                    -0.0428773, -0.0173679, 0.00495183;...
                    0.0427001, -0.00407746, 0.0180194;...
                    0.0463395, 0.0073981, -0.000798676;...
                    -0.0144577, 0.00142989, 0.000313055;...
                    0.0106463, -0.00848233, -0.0111967;...
                    -0.0116988, -0.00871607, -0.0106512;...
                    0.0139006, 0.010341, -0.00522298;...
                    -0.0108479, -0.0275267, -0.00740473;...
                    -0.00850456, 0.0117589, -0.011563;...
                    -0.0136809, -0.0125035, -0.0012194;...
                    -0.00114387, 0.0180089, 0.00564473;...
                    0.00884172, -0.000508907, -0.0162453;...
                    0.00453732, 0.0116105, -0.0142162;...
                    -0.0183339, -0.00421451, -0.00189666;...
                    -0.016628, 0.00771699, -0.00463138;...
                    0.0111267, 0.0149283, -0.00329158;...
                    -0.00933691, 0.0122522, 0.0110972;...
                    -0.0172008, -0.00432714, 0.00768363;...
                    0.0162031, 0.00985375, 0.00110469;...
                    -0.0223897, 0.00957707, -0.0155841;...
                    0.0133594, -0.00464802, -0.0134327;...
                    0.0226299, 0.0291718, 0.0284566;...
                    -0.0254911, 0.000959356, -0.0167186;...
                    -0.00208515, 0.0355851, 0.031197;...
                    0.00384799, 0.0395279, 0.0257932;...
                    0.040523, 0.0237627, 0.00609188;...
                    -8.11857e-05, -0.0446407, 0.018215;...
                    -0.00323668, 0.0254457, -0.0408289;...
                    -0.00854545, -0.0289668, -0.00425729;...
                    -0.0370871, -0.00535246, 0.0304274;...
                    -0.0101852, -0.0264369, -0.0390986;...
                    -0.0107656, 0.00827356, -0.0462433;...
                    -0.0478224, -0.00169006, 0.00589464;...
                    -0.00534411, -0.0298304, 0.037499;...
                    0.00592314, 0.0417761, 0.0233293;...
                    -0.0471877, -0.00452045, 0.00880196;...
                    0.0266519, -0.039122, -0.00915063;...
                    -0.00154947, 0.0419799, 0.022711;...
                    -0.0264685, 0.0386408, -0.00984419;...
                    -0.000439045, 0.0433647, -0.0202115;...
                    -0.0398179, 0.00438145, 0.0268314;...
                    0.0112266, -0.0289406, 0.0368916;...
                    0.0279799, 0.0336018, 0.019463;...
                    -0.0270711, 0.00867085, -0.0110544;...
                    0.0416234, 0.0240531, 0.00737646;...
                    0.0422535, 0.0257359, -0.000777661;...
                    0.0143163, 0.0239588, 0.0413675;...
                    -0.036784, 0.0239841, 0.0237054;...
                    -0.0391537, -0.00739415, -0.0300427;...
                    -0.0107459, 0.0129261, 0.0469861;...
                    -0.0233668, -0.0105952, -0.0428017;...
                    -0.0381503, -0.0225022, -0.0229881;...
                    0.0488359, -0.0101988, 0.00113688;...
                    0.0187415, 0.00406487, -0.00304963;...
                    -0.010438, 0.0237583, -0.0160255;...
                    -0.014599, -0.00741349, -0.00135217;...
                    -0.00476097, 0.00396419, 0.0152164;...
                    -0.0160091, -0.00234944, 0.00152137;...
                    -0.0178912, -0.00750316, -0.000820839;...
                    -0.0145432, 0.00744047, 0.0103939;...
                    -0.00765749, 0.0177828, 0.000430018;...
                    0.0136106, 0.0129847, -0.00481804;...
                    0.0375709, 0.0323437, 0.0190242;...
                    0.0503459, -0.000878584, 0.0107599;...
                    0.0144839, 0.0123661, 0.00523669;...
                    -0.00552327, 0.0103862, 0.0526804;...
                    -0.00878237, 0.0191141, -0.0510261;...
                    -0.034865, 0.00671751, 0.0428954;...
                    -0.0132968, -0.0108568, 0.0252102;...
                    -0.0193445, -0.0378056, 0.0312241;...
                    0.0056531, -0.0313903, -0.0429107;...
                    0.00576421, 0.0033653, 0.0036152;...
                    -0.008184, -0.0276113, -0.0100431;...
                    0.0450903, -0.01759, 0.0186437;...
                    0.00237997, 0.035476, -0.0376341;...
                    -0.0412444, -0.0214395, 0.022934;...
                    -0.0101844, 0.00671525, 0.0155342;...
                    -0.00724273, -0.017343, -0.00607392;...
                    0.00140105, -0.0197006, 0.000230149;...
                    0.0116209, -0.011557, -0.0110235;...
                    0.017032, -0.00528021, -0.00849468;...
                    0.0230318, -0.0216276, 0.0432424;...
                    -0.00385239, -0.0331119, 0.0418029;...
                    0.0424507, 0.0280039, -0.0178795;...
                    0.0300364, -0.0388944, 0.0212876;...
                    0.00842593, 0.0418267, -0.0323675;...
                    0.000321489, 0.0182823, -0.0498873;...
                    0.0322888, 0.0336965, 0.0261028;...
                    -0.0360413, -0.0395377, 2.40059e-06;...
                    0.0344831, -0.0392505, -0.00209968;...
                    0.0306191, 0.00421694, 0.0426065;...
                    -0.0354668, -0.0366632, 0.0114895;...
                    -0.0401689, 0.0303138, 0.012185;...
                    -0.0350874, -0.00948671, 0.0375894;...
                    -0.0409263, 0.00330942, 0.0316893;...
                    0.0217305, 0.0472616, 0.00807746;...
                    -0.0239333, 0.0329583, -0.0334064;...
                    0.0444085, 0.00739046, -0.0274167;...
                    0.0243186, -0.0229507, 0.0407466;...
                    0.0125018, -0.0123791, -0.0496878;...
                    0.0329891, -0.0410974, -0.00106461;...
                    0.0389274, -0.00612693, -0.0350075;...
                    -0.0121221, 0.00921621, 0.0504631;...
                    0.0465305, -0.0102232, 0.0223927;...
                    0.0525476, 0.00279595, 4.96356e-05;...
                    -0.0465005, -0.0237203, -0.00302349;...
                    -0.00352381, -0.00820751, -0.0166124;...
                    -0.013368, -0.0114083, 0.00686694;...
                    0.00816464, 0.00114099, -0.018714;...
                    -0.00510025, 0.00921718, -0.0175273;...
                    0.0165583, 0.0119672, 0.000885451;...
                    -0.0120398, 0.00109206, -0.0164932;...
                    0.0156663, 0.00113166, 0.0129555;...
                    0.0153538, -0.00307577, 0.0128747;...
                    0.000322836, -0.0273133, 0.0135686;...
                    -0.00454595, 0.0138725, -0.0143201;...
                    0.0189283, 0.00415739, 0.00652765;...
                    -0.0143946, -0.0145049, 0.000760509;...
                    0.00581154, 0.00800421, 0.0288511;...
                    0.0177041, 0.0082224, 0.00609356;...
                    0.0081903, -0.0141499, 0.0122831;...
                    0.0023455, 0.0200486, -0.00187323;...
                    -0.0155745, 0.0129601, -0.0027651;...
                    -0.00750544, 0.013807, 0.0134642;...
                    -0.0115047, 2.40371e-05, -0.0173255;...
                    0.000773467, 0.0207106, -0.0223763;...
                    -0.0157639, 0.0011137, -0.0136337;...
                    0.0483275, 0.0187943, -0.0115895;...
                    0.0310753, 0.0248469, 0.0352142;...
                    -0.0383689, -0.01769, 0.0322485;...
                    0.0251729, -0.0024906, -0.0466452;...
                    -0.0512654, 0.00612182, -0.0106204;...
                    -0.0186208, -0.0111119, 0.00116725;...
                    0.00755037, -0.0522803, -0.00891403;...
                    0.0478815, 0.0158935, 0.0166688;...
                    -0.0417311, 0.0299445, 0.0135992;...
                    0.0135141, -0.0467875, -0.0222805;...
                    -0.000835578, 0.0222815, -0.00307686;...
                    -0.000733095, 0.00541777, -0.0214521;...
                    0.000269227, 0.00552885, 0.0213967;...
                    -0.00555224, -0.0152246, -0.0144553;...
                    -0.0133769, -0.000661719, -0.0171172;...
                    0.00730105, -0.0201212, -0.00564914;...
                    0.0123837, -0.0129434, -0.0135673;...
                    -0.0130728, 0.0178599, -0.000456586;...
                    -0.0177693, -0.0124237, -0.00734908;...
                    -0.00133543, -0.0229411, 0.048374;...
                    0.0165006, 0.0238517, 0.0450216;...
                    -0.0086685, -0.0223015, -0.0479126;...
                    0.0384418, 0.0272264, 0.0254769;...
                    -0.011042, 0.0246972, -0.0140835;...
                    0.0222753, -0.00411391, -0.00353825;...
                    -0.000313581, -0.0228231, 0.00219542;...
                    -0.0100291, -0.00305252, 0.0199762;...
                    0.00388306, 0.0222785, -0.00377171;...
                    0.00468763, -0.00364435, 0.022221;...
                    -0.013643, 0.016086, -0.00912593;...
                    0.000425614, 0.048644, 0.0213699;...
                    0.0472975, 0.0174702, 0.017787;...
                    0.0521813, -0.00910572, 0.00415779;...
                    0.0318797, -0.0369269, 0.0210518;...
                    0.0157025, 0.0155306, 0.00780108;...
                    0.0118953, -0.0132951, -0.015007;...
                    0.0138896, 0.00705737, -0.0173461;...
                    -0.00138523, -0.000148994, 0.0229398;...
                    -0.00749762, 0.0156028, -0.0151166;...
                    0.00694252, -0.0042984, -0.0214908;...
                    -0.00169973, -0.00690882, -0.000877451;...
                    0.00801852, 0.0179816, 0.0232936;...
                    -0.0151171, 0.0159013, 0.00788881;...
                    0.00262029, -0.0181912, -0.0144903;...
                    -0.0180037, -0.0141651, -0.00622343;...
                    -0.00728038, 0.052041, -0.00414032;...
                    -0.0140873, -0.00210062, 0.0510961;...
                    0.0382368, -0.00207563, -0.0362222;...
                    0.0222996, -0.0296043, 0.0374797;...
                    -0.00752182, 0.0498892, 0.0152613;...
                    0.0495979, -0.0121789, -0.0130435;...
                    0.0361309, 0.0202807, -0.0325832;...
                    0.051387, 0.00897647, -0.00756366;...
                    0.0239307, -0.046927, -0.00189526;...
                    -0.0298758, -0.0169175, -0.0399956;...
                    0.00917225, 0.050953, 0.00942013;...
                    0.0254696, 0.0355732, -0.0286345;...
                    -0.0379455, 0.0338915, -0.0120662;...
                    0.0259433, -0.0445614, 0.00867847;...
                    0.0359169, 0.0278084, -0.0259738;...
                    -0.0207428, -0.0427031, -0.0219151;...
                    0.0231039, 0.0452809, 0.012245;...
                    0.0331155, 0.0321705, -0.0245462;...
                    -0.0290405, 0.0272032, 0.0339223;...
                    0.0136214, -0.000878066, 0.0504755;...
                    0.0245914, -0.0393645, 0.0240787;...
                    -0.00247152, 0.0496127, -0.0163269;...
                    -0.0027464, 0.0514709, 0.00879173;...
                    -0.0159029, 0.0285113, -0.0412946;...
                    -0.0170281, -0.0252723, 0.0424905;...
                    -0.0325898, 0.00978061, 0.0397031;...
                    -0.00583776, -0.0396174, 0.0329622;...
                    -0.0248508, 0.00998185, -0.0145954;...
                    0.00448132, 0.0515692, 0.00994716;...
                    0.0131891, 0.00842452, -0.0503745;...
                    0.0284279, 0.0436827, -0.00787926;...
                    0.0129413, 0.0470972, -0.0197311;...
                    -0.0358339, 0.0148965, -0.0356998;...
                    0.0311531, -0.0252, -0.0335937 ];
                
                h_sos.name              = 'combined_3D_400';
                h_sos.Pdist_decorr      = 10;
                h_sos.dist              = single( D );
                h_sos.acf               = single( R );
                h_sos.sos_freq          = single( F );
                h_sos.sos_amp           = single( 2 / size(F,1) );
                h_sos.init;
                
                if exist( 'dist_decorr','var' )
                    h_sos.dist_decorr = dist_decorr;
                end
                
                if exist( 'distribution','var' )
                    h_sos.distribution = distribution;
                end
            end
        end
        
        % Get functions
        function out = get.dist_decorr(h_sos)
            out = h_sos.Pdist_decorr;
        end
        function out = get.dimensions(h_sos)
            out = size( h_sos.sos_freq,2 );
        end
        function out = get.no_coefficients(h_sos)
            out = size( h_sos.sos_freq,1 );
        end
        
        % Set functions
        function set.dist_decorr(h_sos,val)
            if ~( all(size(val) == [1 1]) && isnumeric(val) && isreal(val) && val > 0 )
                error('QuaDRiGa:qd_sos:wrongInputValue','??? "decorr_dist" must be scalar and > 0')
            end
            if isempty( h_sos.dist )
                error('QuaDRiGa:qd_sos:noACFgiven','??? No ACF specified')
            end
            
            % Adjust distance vector
            maxD_old = h_sos.dist(end);
            h_sos.dist = val / h_sos.Pdist_decorr .* h_sos.dist;
            
            % Adjust the frequencies
            maxD_new = max(h_sos.dist);
            h_sos.sos_freq = h_sos.sos_freq * maxD_old / maxD_new;
            
            % Save new decorr distance
            h_sos.Pdist_decorr = val;
        end
        
        function set.distribution(h_sos,value)
            supported_types = {'Normal','Uniform'};
            if ~( ischar(value) && any( strcmpi(value,supported_types)) )
                str = 'Distribution not supported; supported types are: ';
                no = numel(supported_types);
                for n = 1:no
                    str = [str,supported_types{n}];
                    if n<no
                        str = [str,', '];
                    end
                end
                error('QuaDRiGa:qd_sos:wrongDistributionType',str);
            end
            h_sos.distribution = value;
        end
    end
    
    methods(Static)
        h_sos = generate( R, L, D, dim, uniform_smp, debug )
        h_sos = load( filename )
        [ s, h_sos ] = randn( dist_decorr, coord )
        [ s, h_sos ] = rand( dist_decorr, coord )
        [ s, h_sos ] = randi( dist_decorr, coord, imax )
    end
end
