function [ par, h_builder ] = gen_lsf_parameters( h_layout, overlap, usage, check_parfiles )
%GEN_LSF_PARAMETERS Generates large scale parameters and stores them in 'qd_layout.track.par'
%
% Calling object:
%   Single object
%
% Description:
%   Normally, parameters are handled by objects of the 'qd_builder' class, which are generated by
%   calling 'layout.init_builder'. However, this method is rather inflexible when the user wants to
%   manipulate the parameters directly. As an alternative, parameters can be provided in the
%   property 'qd_layout.track.par' of the 'qd_track' class. This allows the user to edit parameters
%   without dealing with the 'qd_builder' objects. This function extracts the LSP for the  given
%   scenario from the 'qd_builder' class and stores them in 'qd_layout.track.par'. Hence, it
%   automatically generates the LSP and, thus, implements an easy-to-use interface for the
%   'qd_builder' class.
%
% Input:
%   overlap
%   The length of the overlapping part relative to the segment length. When there are scenario
%   transitions, KF and PG change smoothly during a predefined interval. The length of that
%   interval is a percentage of the previous segment. The parameter overlap adjusts this percentage,
%   ranging from 0 (i.e. very hard, step-like change at the scenario boundary) to 1 (very smooth,
%   but long transition).
%
%   usage
%   Changes the behavior of the method. Options are
%      * 0 - Deletes all existing parameters from the track.
%      * 1 - Deletes all existing parameters from the track and generates new ones. Existing LSP
%            will be overwritten.
%      * 2 - Keeps existing parameters, but generates missing ones. This is useful when, for
%            example, the effective PG is provided, but the other LSP are unknown. In this case, the
%            unknown "gaps" are filled with values which are generated from the provided scenario 
%            description.  (Default mode)
%
%
%   check_parfiles
%   check_parfiles = 0 / 1 (default: 1) Disables (0) or enables (1) the parsing of shortnames and
%   the validity-check for the config-files. This is useful, if you know that the parameters in the
%   files are valid. In this case, this saves execution time.
%
% Output:
%   par
%   The automatically generated parameters. This cell array contains a parameter structure of the
%   LSP for each receiver with the following fields:
%      * ds - The delay spread in [s] per segment
%      * kf - The Ricean K-Factor in [dB] per snapshot
%      * pg - The effective path gain in [dB] excluding antenna gains per snapshot
%      * asD - The azimuth angle spread in [deg] per segment at the transmitter
%      * asA - The azimuth angle spread in [deg] per segment at the receiver
%      * esD - The elevation angle spread in [deg] per segment at the transmitter
%      * esA - The elevation angle spread in [deg] per segment at the receiver
%      * xpr - The NLOS cross-polarization in [dB] per segment
%
%    An identical copy of this variable is assigned to 'track.par'.
%
%   h_builder
%   An array of 'qd_builder' objects. Rows correspond to the scenarios, columns correspond to the
%   transmitters.
%
%
% QuaDRiGa Copyright (C) 2011-2017 Fraunhofer Heinrich Hertz Institute
% e-mail: quadriga@hhi.fraunhofer.de
%
% QuaDRiGa is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published
% by the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.


if numel( h_layout ) > 1 
   error('QuaDRiGa:qd_layout:set_scenario','set_scenario not definded for object arrays.');
else
    h_layout = h_layout(1,1); % workaround for octave
end

% Parse the input variables
if exist( 'overlap' , 'var' ) && ~isempty( overlap )
    if ~( isnumeric(overlap) && all(size(overlap) == [1 1]) && isreal(overlap) ...
            && overlap<=1 && overlap>=0 )
        error('??? "overlap" must be scalar, and in between 0 and 1')
    end
else
    overlap = 0.5;
end

if exist( 'usage' , 'var' ) && ~isempty( usage )
    usage = real( usage(1) );
else
    usage = 2;
end

if exist( 'check_parfiles' , 'var' ) && ~isempty( check_parfiles )
    if ~( all(size(check_parfiles) == [1 1]) ...
            && (isnumeric(check_parfiles) || islogical(check_parfiles)) ...
            && any( check_parfiles == [0 1] ) )
        error('??? "check_parfiles" must be 0 or 1')
    end
else
    check_parfiles = true;
end

% Reset the pairing matrix. We calculate a set of parameters for each track
% and for each Tx in the layout, regardless of it being needed or not.
pairing = h_layout.pairing;
h_layout.set_pairing;

% Generate large scale parameters using the correlation maps
switch usage
    case 0  % Delete all existing parameters, but keep O2I parameters
        for n = 1 : h_layout.no_rx
            par = h_layout.track(1,n).par;
            if ~isempty( par )
                if isempty( par.o2i_loss )
                    h_layout.track(1,n).par_nocheck = [];
                else
                    fn = fieldnames( par );
                    for m = 1 : numel(fn)
                       if ~strcmp( fn{m} , 'o2i_loss' ) && ~strcmp( fn{m} , 'o2i_d3din' )
                           par.( fn{m} ) = [];
                       end
                    end
                    h_layout.track(1,n).par_nocheck = par;
                end
            end
        end
        
    case 1  % Overwrite existing parameters
        h_builder = h_layout.init_builder( check_parfiles );
        gen_lsf_parameters( h_builder, 1 );   % Force new parameters
        
    case 2  % Generate only non-existing parameters
        h_builder = h_layout.init_builder( check_parfiles );
        gen_lsf_parameters( h_builder, 0 );
        
    otherwise
        error('??? "usage" must be 0, 1 or 2');
end

if usage > 0
    
    % Replace the scenario shortnames with their long form
    [ sup_scenarios , file_names ] = qd_builder.supported_scenarios( check_parfiles );
    
    % Extract the positions from h_builder.
    % Those are needed for matching the parameters to segments.
    sic = size( h_builder );
    pos = cell( sic );
    o_builder = cell( sic );
    for n = 1 : numel( h_builder )
        [ i1,i2,i3,i4 ] = qf.qind2sub( sic, n );
        pos{ i1,i2,i3,i4 } = h_builder( i1,i2,i3,i4  ).rx_positions;
        o_builder{ i1,i2,i3,i4 } = ones(1,h_builder( i1,i2,i3,i4 ).no_rx_positions);
    end
    
    % Extract the list of scenarios from h_builder.
    scenarios = cell( 1,sic(1) );
    for n = 1 : sic(1)
        scenarios{1,n} = h_builder(n,1).scenario;
    end
    
    % Calculate the number of tracks( n_rx), the number of transmitters (n_tx).
    n_trk  = h_layout.no_rx;
    n_tx   = h_layout.no_tx;
    n_freq = numel( h_layout.simpar.center_frequency );
    par = cell(1,n_trk);
    
    oF = ones( n_freq, 1 );
    
    % Do the calculations for each Rx (i.e. for each track in the layout.)
    for i_trk = 1 : n_trk
        
        % Extract tracks and subtracks for faster processing.
        trk = h_layout.track(1,i_trk);
        subtrk = get_subtrack( trk );
        
        % Copy the o2i parameters
        if ~isempty( h_layout.track(1,i_trk).par )
            o2i_loss  = h_layout.track(1,i_trk).par.o2i_loss;
            o2i_d3din = h_layout.track(1,i_trk).par.o2i_d3din;
        else
            o2i_loss  = [];
            o2i_d3din = [];
        end
        
        % Some temporary variables
        n_seg  = trk.no_segments;
        n_snap = trk.no_snapshots;
        z_seg  = zeros( n_tx, n_seg, n_freq );
        z_snap = zeros( n_tx, n_snap, n_freq );
        
        data = struct('ds',z_seg,'kf',z_snap,'pg',z_snap,...
            'asD',z_seg,'asA',z_seg,'esD',z_seg,'esA',z_seg,'xpr',z_seg,...
            'o2i_loss',o2i_loss,'o2i_d3din',o2i_d3din);
        
        % Read scenario ids from the track
        scen = trk.scenario;
        if size(scen,1) < n_tx
            scen( 2:n_tx,: ) = scen( ones(1,n_tx-1) , : );
        end
        pos_trk = trk.positions_abs;
        
        % Replace the scenario shortnames with their long form
        for i_scen = 1:numel( scen )
            ind = strcmp( scen(i_scen) , sup_scenarios );
            if any(ind)
                scen{i_scen} = file_names{ind}(1:end-5);
            else
                i1 = regexp( scen{i_scen} , '.conf', 'once' );
                i2 = regexp( scen{i_scen} , filesep);
                
                if ~isempty( i1 )
                    if isempty( i2 )
                        scen{i_scen} = scen{i_scen}(1:i1-1);
                    else
                        scen{i_scen} = scen{i_scen}(i2(end)+1:i1-1);
                    end
                end
            end
        end
        
        % Check if the track is closed.
        closed = trk.closed;
        
        % Placeholder for the KF and the PG
        pg = cell( n_tx,n_seg );
        kf = cell( n_tx,n_seg );
        
        % Read the initial segment positions
        pos_seg = zeros( 3, n_seg );
        for i_seg = 1:n_seg
            pos_seg(:,i_seg) = pos_trk( : , trk.segment_index( i_seg ) );
        end
        
        % Do for each segment of the track
        for i_seg = 1 : n_seg
            
            % === Determine the overlapping area ===
            
            % This determines the the start and end-segments for closed tracks
            domerge = true;
            if i_seg < n_seg
                i_seg2 = i_seg + 1;
            elseif n_seg > 1 && closed
                i_seg2 = 1 ;
            else
                i_seg2 = [];
                domerge = false;
            end
            
            % Determine the start and end-points of overlapping and exclusive
            % parts.
            if domerge
                rangem = zeros(3,2);   % The overlapping start / end points
                rangec = zeros(2,2);   % The non-overlapping start / end points
                
                % Determine the end index of the second segment
                rangem(2,2) = subtrk( 1,i_seg2 ).segment_index(end)-1;
                
                % Determine the start index of the second segment
                rangem(2,1) = floor( rangem(2,2) - (rangem(2,2)-2) * overlap );
                
                % Number of overlapping snapshots
                S = rangem(2,2) - rangem(2,1) + 1;
                
                % Determine the end index of the first segment
                rangem(1,2) = subtrk( 1,i_seg ).no_snapshots;
                
                % Determine the start index of the first segment
                rangem(1,1) = rangem(1,2) - S + 1;
                
                % Determine the the non-overlapping area of the first segment
                rangec(1,:) = [ subtrk(1,i_seg).segment_index(end) ,rangem(1,1)-1 ];
                
                % Determine the indexes of the non-overlapping part in the output channel
                if i_seg == 1
                    rangec(2,1) = 1;
                else
                    segments_before = 1:(i_seg-1);
                    
                    tmp = zeros( numel(segments_before),1 );
                    for i_tmp = 1:numel(segments_before)
                        tmp( i_tmp ) = subtrk( 1,segments_before(i_tmp) ).segment_index(end);
                    end
                    
                    rangec(2,1) = sum( cat(1,subtrk( 1,segments_before ).no_snapshots) -tmp+1 ) + 1;
                end
                rangec(2,2) = rangec(1,2) - rangec(1,1) + rangec(2,1);
                
                % Determine the indexes of the overlapping part in the output channel
                rangem(3,:) = [rangec(2,2)+1, rangec(2,2)+S];
                
                % Extend the range of the data structure
                rangem = [rangem(1,1):rangem(1,2) ; rangem(2,1):rangem(2,2) ; rangem(3,1):rangem(3,2) ];
                rangec = [rangec(1,1):rangec(1,2) ; rangec(2,1):rangec(2,2)];
                
                % Calculate the ramp
                tmp  = (1:S)/(S+1);
                ramp = 0.5*(1+sin((tmp-0.5)*pi));
                
            else
                % Here, we only have an exclusive part without overlapping
                rangec = zeros(2,2);   % The non-overlapping start / end points
                
                % Determine the the non-overlapping area of the segment
                rangec(1,:) = [ subtrk(1,i_seg).segment_index(end) ,...
                    subtrk(1,i_seg).no_snapshots  ];
                
                if i_seg > 1
                    segments_before = 1:(i_seg-1);
                    
                    tmp = zeros( numel(segments_before),1 );
                    for i_tmp = 1:numel(segments_before)
                        tmp( i_tmp ) = subtrk( 1,segments_before(i_tmp) ).segment_index(end);
                    end
                    
                    rangec(2,1) = sum( cat(1,subtrk( 1,segments_before ).no_snapshots) -tmp+1 ) + 1;
                    rangec(2,2) = rangec(1,2) - rangec(1,1) + rangec(2,1);
                else
                    rangec(2,1) = rangec(1,1);
                    rangec(2,2) = rangec(1,2);
                end
                
                % Extend the range of the data structure
                rangec = [rangec(1,1):rangec(1,2) ; rangec(2,1):rangec(2,2)];
            end
            % === End ===
            
            % Do for each transmitter
            % Extract the initial parameters from the parameter_sets
            for i_tx = 1 : n_tx
                
                % Determine the index of the current segment in the
                % parameter_set objects.
                i_scen = strcmp( scen( i_tx , i_seg ) , scenarios );
                i_scen = find( i_scen ); % Needed for octave
                tmp = pos{ i_scen , i_tx } - pos_seg( : , i_seg*o_builder{ i_scen , i_tx } );
                tmp = sum( tmp.^2 , 1 );
                [~,i_par] = min( tmp );
                
                % Extract the SF and KF traces for the current segment
                if isempty( pg{i_tx,i_seg} )
                    evaltrack = h_builder( i_scen,i_tx ).rx_track( 1,i_par );
                    if usage == 1
                        evaltrack.par = [];
                    end
                    [ tmp_sf, tmp_kf ] = h_builder( i_scen,i_tx ).get_sf_profile( evaltrack );
                    
                    if isempty(evaltrack.par) || isempty( evaltrack.par.pg )
                        [ tmp_loss , scale_sf ] = h_builder( i_scen,i_tx ).get_pl( evaltrack );
                    else
                        tmp_loss = 0;
                        scale_sf = 1;
                    end
                    
                    % Merging of channel coefficient id done using the L1-Norm.
                    % We have to do the same here.
                    pg{ i_tx, i_seg } = sqrt( 10.^( 0.1*( -tmp_loss + 10*log10( tmp_sf ) .* scale_sf )));
                    kf{ i_tx, i_seg } = sqrt( tmp_kf );
                end
                
                % Determine the SF and KF traces for the second segment in the
                % overlapping area.
                if domerge
                    i_scen2 = strcmp( scen( i_tx , i_seg2 ) , scenarios );
                    i_scen2 = find( i_scen2 ); % Needed for octave
                    tmp = pos{ i_scen2 , i_tx } -...
                        pos_seg(:,i_seg2*o_builder{ i_scen2 , i_tx } );
                    tmp = sum( tmp.^2 , 1 );
                    [~,i_par2] = min( tmp );
                    
                    if isempty( pg{i_tx,i_seg2} )
                        evaltrack = h_builder( i_scen2,i_tx ).rx_track(1, i_par2 );
                        if usage == 1
                            evaltrack.par = [];
                        end
                        [ tmp_sf, tmp_kf ] = h_builder( i_scen2,i_tx ).get_sf_profile( evaltrack );
                        
                        if isempty(evaltrack.par) || isempty( evaltrack.par.pg )
                            [ tmp_loss , scale_sf ] = h_builder( i_scen2,i_tx ).get_pl( evaltrack );
                        else
                            tmp_loss = 0;
                            scale_sf = 1;
                        end
                        
                        % Merging of channel coefficients in "channel.merge" is
                        % done using the L1-Norm. We have to do the same here.
                        pg{i_tx,i_seg2} = sqrt( 10.^( 0.1*( -tmp_loss + 10*log10( tmp_sf ) .* scale_sf )));
                        kf{ i_tx,i_seg2 } = sqrt( tmp_kf );
                    end
                end
                
                % Copy the segment-parameters from the parameter_set objects.
                data.ds(  i_tx,i_seg,: ) = permute( h_builder( i_scen,i_tx ).ds( :,i_par ) , [3,2,1] );
                data.asD( i_tx,i_seg,: ) = permute( h_builder( i_scen,i_tx ).asD( :,i_par ), [3,2,1] );
                data.asA( i_tx,i_seg,: ) = permute( h_builder( i_scen,i_tx ).asA( :,i_par ), [3,2,1] );
                data.esD( i_tx,i_seg,: ) = permute( h_builder( i_scen,i_tx ).esD( :,i_par ), [3,2,1] );
                data.esA( i_tx,i_seg,: ) = permute( h_builder( i_scen,i_tx ).esA( :,i_par ), [3,2,1] );
                data.xpr( i_tx,i_seg,: ) = permute( 10*log10( h_builder( i_scen,i_tx ).xpr( :,i_par ) ), [3,2,1] );
                
                % Extract SF and KF parts
                % This requires merging the overlapping parts from two
                % successive segments.
                
                % Copy the exclusive data
                data.pg( i_tx,rangec(2,:),: ) = permute( pg{i_tx,i_seg}( :,rangec(1,:) ), [3,2,1] );
                data.kf( i_tx,rangec(2,:),: ) = permute( kf{i_tx,i_seg}( :,rangec(1,:) ), [3,2,1] );
                
                % Merge overlapping parts
                if domerge
                    
                    % Merge the SF
                    A = pg{i_tx,i_seg}( :,rangem(1,:) );
                    B = pg{i_tx,i_seg2}( :,rangem(2,:) );
                    data.pg( i_tx,rangem(3,:),: ) = permute( A .* (oF*(1-ramp)) + B .* (oF*ramp), [3,2,1] );
                    
                    % Merge the KF
                    A = kf{i_tx,i_seg}( :,rangem(1,:) );
                    B = kf{i_tx,i_seg2}( :,rangem(2,:) );
                    data.kf( i_tx,rangem(3,:),: ) = permute( A .* (oF*(1-ramp)) + B .* (oF*ramp), [3,2,1] );
                    
                end
            end
        end
        
        % Transform the output into log-values
        data.pg = 10*log10( data.pg.^2 );
        data.kf = 10*log10( data.kf.^2 );
        
        % Processing of subtracks ignores the last value in case of a
        % closed track. This is fixed here.
        if closed
            data.pg( :,end,: ) = data.pg( :,1,: );
            data.kf( :,end,: ) = data.kf( :,1,: );
        end
        
        % Write the data to the track object
        par{i_trk} = data;
        h_layout.track(1,i_trk).par = data;
    end
end

% Set the pairing matrix back to its original values
h_layout.pairing = pairing;

end
